<?php

namespace Tests\Unit;

use App\Support\RetryHelper;
use PHPUnit\Framework\TestCase;

class RetryHelperTest extends TestCase
{
    /** @test */
    public function it_executes_callback_successfully_on_first_try()
    {
        $callCount = 0;

        $result = RetryHelper::retry(function () use (&$callCount) {
            $callCount++;
            return 'success';
        });

        $this->assertEquals('success', $result);
        $this->assertEquals(1, $callCount);
    }

    /** @test */
    public function it_retries_on_failure_and_succeeds()
    {
        $callCount = 0;

        $result = RetryHelper::retry(function () use (&$callCount) {
            $callCount++;
            if ($callCount < 3) {
                throw new \Exception('Temporary failure');
            }
            return 'success after retries';
        }, maxAttempts: 3);

        $this->assertEquals('success after retries', $result);
        $this->assertEquals(3, $callCount);
    }

    /** @test */
    public function it_throws_exception_after_max_retries()
    {
        $callCount = 0;

        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('Persistent failure');

        try {
            RetryHelper::retry(function () use (&$callCount) {
                $callCount++;
                throw new \Exception('Persistent failure');
            }, maxAttempts: 3, baseDelayMs: 1); // Use minimal delay for tests
        } finally {
            $this->assertEquals(3, $callCount);
        }
    }

    /** @test */
    public function it_calculates_exponential_backoff_delay()
    {
        // Attempt 1: 1000ms
        $delay1 = RetryHelper::calculateDelay(1, 1000, 10000);
        $this->assertGreaterThanOrEqual(1000, $delay1);
        $this->assertLessThanOrEqual(1250, $delay1); // Max 25% jitter

        // Attempt 2: 2000ms
        $delay2 = RetryHelper::calculateDelay(2, 1000, 10000);
        $this->assertGreaterThanOrEqual(2000, $delay2);
        $this->assertLessThanOrEqual(2500, $delay2);

        // Attempt 3: 4000ms
        $delay3 = RetryHelper::calculateDelay(3, 1000, 10000);
        $this->assertGreaterThanOrEqual(4000, $delay3);
        $this->assertLessThanOrEqual(5000, $delay3);
    }

    /** @test */
    public function it_respects_max_delay_cap()
    {
        // With base 1000ms and max 5000ms, attempt 5 would be 16000ms but capped at 5000ms
        $delay = RetryHelper::calculateDelay(5, 1000, 5000);

        $this->assertLessThanOrEqual(6250, $delay); // 5000 + 25% jitter max
    }

    /** @test */
    public function it_retries_on_timeout_errors()
    {
        $callCount = 0;

        $result = RetryHelper::retry(function () use (&$callCount) {
            $callCount++;
            if ($callCount < 2) {
                throw new \Exception('Connection timed out');
            }
            return 'recovered';
        }, maxAttempts: 3, baseDelayMs: 1);

        $this->assertEquals('recovered', $result);
        $this->assertEquals(2, $callCount);
    }

    /** @test */
    public function it_retries_on_rate_limit_errors()
    {
        $callCount = 0;

        $result = RetryHelper::retry(function () use (&$callCount) {
            $callCount++;
            if ($callCount < 2) {
                throw new \Exception('Error 429: Too many requests');
            }
            return 'recovered';
        }, maxAttempts: 3, baseDelayMs: 1);

        $this->assertEquals('recovered', $result);
        $this->assertEquals(2, $callCount);
    }

    /** @test */
    public function it_retries_on_server_errors()
    {
        $callCount = 0;

        $result = RetryHelper::retry(function () use (&$callCount) {
            $callCount++;
            if ($callCount < 2) {
                throw new \Exception('HTTP 503 Service Unavailable');
            }
            return 'recovered';
        }, maxAttempts: 3, baseDelayMs: 1);

        $this->assertEquals('recovered', $result);
        $this->assertEquals(2, $callCount);
    }

    /** @test */
    public function it_uses_default_values()
    {
        $this->assertEquals(3, RetryHelper::DEFAULT_MAX_ATTEMPTS);
        $this->assertEquals(1000, RetryHelper::DEFAULT_BASE_DELAY_MS);
        $this->assertEquals(10000, RetryHelper::DEFAULT_MAX_DELAY_MS);
    }
}
