<?php

use App\Http\Controllers\AIController;
use App\Http\Controllers\EmailController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\UserManagementController;
use App\Http\Controllers\ReviewController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

Route::get('/', function () {
    return view('landing');
});

Route::get('/home', function () {
    return redirect()->route('login');
});

// Test route to verify role-based navigation
Route::get('/test-navigation', function () {
    if (!auth()->check()) {
        return 'Not logged in';
    }
    
    $user = auth()->user();
    return [
        'user_role' => $user->role,
        'is_super_admin' => $user->isSuperAdmin(),
        'is_tenant_admin' => $user->isTenantAdmin(),
        'is_agent' => $user->isAgent(),
        'email_index_route' => \App\Helpers\RoleBasedNavigation::getEmailIndexRoute(),
        'bulk_delete_route' => \App\Helpers\RoleBasedNavigation::getBulkDeleteRoute(),
    ];
})->middleware('auth');

// Authentication routes (Breeze)
require __DIR__.'/auth.php';

// Protected routes
Route::middleware(['auth'])->group(function () {
    // Dashboard
    Route::get('/dashboard', function () {
        return view('dashboard');
    })->name('dashboard');

    // Profile routes
    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::put('/profile', [ProfileController::class, 'update'])->name('profile.update');

    // Gmail OAuth routes (using Gmail module)
    Route::get('/oauth/google/start', [App\Modules\Gmail\Controllers\GmailController::class, 'startOAuth'])->name('oauth.google.start');
    Route::get('/oauth/google/callback', [App\Modules\Gmail\Controllers\GmailController::class, 'handleCallback'])->name('oauth.google.callback');

    // AI Provider routes
    Route::get('/ai/settings', [AIController::class, 'settings'])->name('ai.settings');
    Route::get('/ai/providers', [AIController::class, 'getProviders'])->name('ai.providers');
    Route::post('/ai/test-provider', [AIController::class, 'testProvider'])->name('ai.test-provider');
    Route::get('/ai/test-all-providers', [AIController::class, 'testAllProviders'])->name('ai.test-all-providers');
    Route::post('/ai/settings', [AIController::class, 'updateSettings'])->name('ai.update-settings');
    
    // User Management routes (for all authenticated users)
    Route::get('/user-management', [UserManagementController::class, 'index'])->name('user-management.index');
    Route::get('/user-management/create', [UserManagementController::class, 'create'])->name('user-management.create');
    Route::post('/user-management', [UserManagementController::class, 'store'])->name('user-management.store');
    Route::get('/user-management/{user}/edit', [UserManagementController::class, 'edit'])->name('user-management.edit');
    Route::put('/user-management/{user}', [UserManagementController::class, 'update'])->name('user-management.update');
    Route::delete('/user-management/{user}', [UserManagementController::class, 'destroy'])->name('user-management.destroy');
});

// Super Admin routes
Route::middleware(['auth', 'super-admin'])->prefix('super-admin')->name('super-admin.')->group(function () {
    // Email routes
    Route::get('/emails', [App\Http\Controllers\SuperAdmin\EmailController::class, 'index'])->name('email.index');
    Route::post('/emails/bulk-delete', [App\Http\Controllers\SuperAdmin\EmailController::class, 'bulkDelete'])->name('email.bulk-delete');
    Route::delete('/emails/{thread}', [App\Http\Controllers\SuperAdmin\EmailController::class, 'destroyThread'])->name('email.thread.delete');
    
    // Review routes
    Route::get('/reviews', [App\Http\Controllers\SuperAdmin\ReviewController::class, 'index'])->name('reviews.index');
    Route::post('/reviews/import-locations', [App\Http\Controllers\SuperAdmin\ReviewController::class, 'importLocations'])->name('reviews.import-locations');
    Route::post('/reviews/ingest', [App\Http\Controllers\SuperAdmin\ReviewController::class, 'ingest'])->name('reviews.ingest');
    Route::post('/reviews/disconnect', [App\Http\Controllers\SuperAdmin\ReviewController::class, 'disconnectGoogle'])->name('reviews.disconnect');
    Route::post('/reviews/{review}/draft', [App\Http\Controllers\SuperAdmin\ReviewController::class, 'draft'])->name('reviews.draft');
    Route::post('/reviews/{review}/approve', [App\Http\Controllers\SuperAdmin\ReviewController::class, 'approve'])->name('reviews.approve');
    Route::delete('/reviews/{review}', [App\Http\Controllers\SuperAdmin\ReviewController::class, 'destroy'])->name('reviews.delete');
    Route::post('/reviews/bulk', [App\Http\Controllers\SuperAdmin\ReviewController::class, 'bulkDelete'])->name('reviews.bulk-delete');
    
    // User Management routes
    Route::get('/user-management', [UserManagementController::class, 'index'])->name('user-management.index');
    Route::get('/user-management/create', [UserManagementController::class, 'create'])->name('user-management.create');
    Route::post('/user-management', [UserManagementController::class, 'store'])->name('user-management.store');
    Route::get('/user-management/{user}/edit', [UserManagementController::class, 'edit'])->name('user-management.edit');
    Route::put('/user-management/{user}', [UserManagementController::class, 'update'])->name('user-management.update');
    Route::delete('/user-management/{user}', [UserManagementController::class, 'destroy'])->name('user-management.destroy');

    // Settings routes
    Route::get('/settings', [App\Http\Controllers\SuperAdmin\SettingsController::class, 'index'])->name('settings.index');
    Route::post('/settings/ingestion-interval', [App\Http\Controllers\SuperAdmin\SettingsController::class, 'updateIngestionInterval'])->name('settings.update-ingestion-interval');
});

// Tenant Admin routes
Route::middleware(['auth', 'tenant-admin'])->prefix('tenant-admin')->name('tenant-admin.')->group(function () {
    // Email routes
    Route::get('/emails', [App\Http\Controllers\TenantAdmin\EmailController::class, 'index'])->name('email.index');
    Route::post('/emails/bulk-delete', [App\Http\Controllers\TenantAdmin\EmailController::class, 'bulkDelete'])->name('email.bulk-delete');
    Route::delete('/emails/{thread}', [App\Http\Controllers\TenantAdmin\EmailController::class, 'destroyThread'])->name('email.thread.delete');
    
    // Review routes
    Route::get('/reviews', [App\Http\Controllers\TenantAdmin\ReviewController::class, 'index'])->name('reviews.index');
    Route::post('/reviews/import-locations', [App\Http\Controllers\TenantAdmin\ReviewController::class, 'importLocations'])->name('reviews.import-locations');
    Route::post('/reviews/ingest', [App\Http\Controllers\TenantAdmin\ReviewController::class, 'ingest'])->name('reviews.ingest');
    Route::post('/reviews/disconnect', [App\Http\Controllers\TenantAdmin\ReviewController::class, 'disconnectGoogle'])->name('reviews.disconnect');
    Route::post('/reviews/{review}/draft', [App\Http\Controllers\TenantAdmin\ReviewController::class, 'draft'])->name('reviews.draft');
    Route::post('/reviews/{review}/approve', [App\Http\Controllers\TenantAdmin\ReviewController::class, 'approve'])->name('reviews.approve');
    Route::delete('/reviews/{review}', [App\Http\Controllers\TenantAdmin\ReviewController::class, 'destroy'])->name('reviews.delete');
    Route::post('/reviews/bulk', [App\Http\Controllers\TenantAdmin\ReviewController::class, 'bulkDelete'])->name('reviews.bulk-delete');
    
    // User Management routes (for tenant users)
    Route::get('/user-management', [UserManagementController::class, 'index'])->name('user-management.index');
    Route::get('/user-management/create', [UserManagementController::class, 'create'])->name('user-management.create');
    Route::post('/user-management', [UserManagementController::class, 'store'])->name('user-management.store');
    Route::get('/user-management/{user}/edit', [UserManagementController::class, 'edit'])->name('user-management.edit');
    Route::put('/user-management/{user}', [UserManagementController::class, 'update'])->name('user-management.update');
    Route::delete('/user-management/{user}', [UserManagementController::class, 'destroy'])->name('user-management.destroy');

    // Settings routes
    Route::get('/settings', [App\Http\Controllers\TenantAdmin\SettingsController::class, 'index'])->name('settings.index');
    Route::post('/settings/ingestion-interval', [App\Http\Controllers\TenantAdmin\SettingsController::class, 'updateIngestionInterval'])->name('settings.update-ingestion-interval');
});

// Agent routes
Route::middleware(['auth', 'agent'])->prefix('agent')->name('agent.')->group(function () {
    // Email routes
    Route::get('/emails', [App\Http\Controllers\Agent\EmailController::class, 'index'])->name('email.index');
    Route::post('/emails/bulk-delete', [App\Http\Controllers\Agent\EmailController::class, 'bulkDelete'])->name('email.bulk-delete');
    Route::delete('/emails/{thread}', [App\Http\Controllers\Agent\EmailController::class, 'destroyThread'])->name('email.thread.delete');
    
    // Review routes
    Route::get('/reviews', [App\Http\Controllers\Agent\ReviewController::class, 'index'])->name('reviews.index');
    Route::post('/reviews/import-locations', [App\Http\Controllers\Agent\ReviewController::class, 'importLocations'])->name('reviews.import-locations');
    Route::post('/reviews/ingest', [App\Http\Controllers\Agent\ReviewController::class, 'ingest'])->name('reviews.ingest');
    Route::post('/reviews/disconnect', [App\Http\Controllers\Agent\ReviewController::class, 'disconnectGoogle'])->name('reviews.disconnect');
    Route::post('/reviews/{review}/draft', [App\Http\Controllers\Agent\ReviewController::class, 'draft'])->name('reviews.draft');
    Route::post('/reviews/{review}/approve', [App\Http\Controllers\Agent\ReviewController::class, 'approve'])->name('reviews.approve');
    Route::delete('/reviews/{review}', [App\Http\Controllers\Agent\ReviewController::class, 'destroy'])->name('reviews.delete');
    Route::post('/reviews/bulk', [App\Http\Controllers\Agent\ReviewController::class, 'bulkDelete'])->name('reviews.bulk-delete');
});

// Legacy routes for backward compatibility (will be removed)
Route::middleware(['auth'])->group(function () {
    // Email routes
    Route::get('/inbox', [EmailController::class, 'index'])->name('email.index');
    Route::post('/inbox/ingest', [EmailController::class, 'ingest'])->name('email.ingest');
    Route::post('/threads/{thread}/draft', [EmailController::class, 'draft'])->name('email.draft');
    Route::get('/drafts/{draft}/edit', [EmailController::class, 'edit'])->name('email.draft.edit');
    Route::put('/drafts/{draft}', [EmailController::class, 'update'])->name('email.draft.update');
    Route::post('/drafts/{draft}/regenerate', [EmailController::class, 'regenerate'])->name('email.draft.regenerate');
    Route::post('/drafts/{draft}/approve', [EmailController::class, 'approveAndSend'])->name('email.approve');
    Route::delete('/drafts/{draft}', [EmailController::class, 'destroy'])->name('email.draft.delete');
    Route::delete('/threads/{thread}', [EmailController::class, 'destroyThread'])->name('email.thread.delete');
    Route::post('/threads/bulk', [EmailController::class, 'bulkDelete'])->name('email.threads.bulk-delete');
    
    // Email account management routes
    Route::get('/email-accounts', [EmailController::class, 'accounts'])->name('email.accounts');
    Route::post('/email-accounts/{mailAccount}/set-primary', [EmailController::class, 'setPrimary'])->name('email.set-primary');
    Route::delete('/email-accounts/{mailAccount}', [EmailController::class, 'deleteAccount'])->name('email.delete-account');
    
    // SMTP test connection route
    Route::post('/smtp-accounts/test-connection', [EmailController::class, 'testSmtpConnection'])->name('smtp.test-connection');
    
    // SMTP+IMAP account routes
    Route::post('/smtp-imap-accounts', [EmailController::class, 'storeSmtpWithImapAccount'])->name('smtp-imap.store');
    Route::post('/imap-accounts/test-connection', [EmailController::class, 'testImapConnection'])->name('imap.test-connection');
    Route::post('/email-accounts/{mailAccount}/update-imap-password', [EmailController::class, 'updateImapPassword'])->name('email.update-imap-password');
    
    // SMTP+IMAP setup page
    Route::get('/email/smtp-imap-setup', function() {
        return view('email.smtp-imap-setup');
    })->name('smtp-imap.setup');
    
    // Add IMAP to existing SMTP account
    Route::post('/email-accounts/{mailAccount}/add-imap', [EmailController::class, 'addImapToExisting'])->name('email.add-imap');
    
    // Email signatures routes
    Route::get('/email-accounts/{mailAccount}/signatures', [EmailController::class, 'signatures'])->name('email.signatures');
    Route::get('/email-accounts/{mailAccount}/signatures/data', [EmailController::class, 'getSignatures'])->name('email.signatures.data');
    Route::post('/email-accounts/{mailAccount}/signatures', [EmailController::class, 'storeSignature'])->name('email.signatures.store');
    Route::put('/email-accounts/{mailAccount}/signatures/{signature}', [EmailController::class, 'updateSignature'])->name('email.signatures.update');
    Route::delete('/email-accounts/{mailAccount}/signatures/{signature}', [EmailController::class, 'deleteSignature'])->name('email.signatures.delete');

    // Reviews (temporary - testing main controller)
    Route::get('/reviews', [ReviewController::class, 'index'])->name('reviews.index');
    Route::post('/reviews/import-locations', [ReviewController::class, 'importLocations'])->name('reviews.import-locations');
    Route::post('/reviews/ingest', [ReviewController::class, 'ingest'])->name('reviews.ingest');
    Route::post('/reviews/disconnect', [ReviewController::class, 'disconnectGoogle'])->name('reviews.disconnect');
    Route::get('/reviews/{review}/edit', [ReviewController::class, 'edit'])->name('reviews.edit');
    Route::put('/reviews/{review}', [ReviewController::class, 'update'])->name('reviews.update');
    Route::delete('/reviews/{review}', [ReviewController::class, 'deleteReview'])->name('reviews.delete');
    Route::post('/reviews/{review}/draft', [ReviewController::class, 'draft'])->name('reviews.draft');
    Route::post('/reviews/{review}/approve', [ReviewController::class, 'approve'])->name('reviews.approve');
    Route::post('/reviews/{review}/post', [ReviewController::class, 'post'])->name('reviews.post');
    Route::delete('/reviews/{reviewDraft}/delete-draft', [ReviewController::class, 'deleteDraft'])->name('reviews.draft.delete');
    Route::post('/reviews/bulk', [ReviewController::class, 'bulkDelete'])->name('reviews.bulk-delete');

    // Agent routes
    Route::prefix('agent')->middleware(['agent'])->group(function () {
        Route::get('/reviews', [App\Http\Controllers\Agent\ReviewController::class, 'index'])->name('agent.reviews.index');
        Route::post('/reviews/import-locations', [App\Http\Controllers\Agent\ReviewController::class, 'importLocations'])->name('agent.reviews.import-locations');
        Route::post('/reviews/ingest', [App\Http\Controllers\Agent\ReviewController::class, 'ingest'])->name('agent.reviews.ingest');
        Route::post('/reviews/{review}/draft', [App\Http\Controllers\Agent\ReviewController::class, 'draft'])->name('agent.reviews.draft');
        Route::post('/reviews/{review}/approve', [App\Http\Controllers\Agent\ReviewController::class, 'approve'])->name('agent.reviews.approve');
        Route::post('/reviews/bulk', [App\Http\Controllers\Agent\ReviewController::class, 'bulkDelete'])->name('agent.reviews.bulk-delete');
        Route::delete('/reviews/{review}', [App\Http\Controllers\Agent\ReviewController::class, 'destroy'])->name('agent.reviews.delete');
        
        // Email routes
        Route::get('/email', [App\Http\Controllers\Agent\EmailController::class, 'index'])->name('agent.email.index');
        Route::get('/email/thread/{thread}', [App\Http\Controllers\Agent\EmailController::class, 'show'])->name('agent.email.show');
        Route::post('/email/bulk', [App\Http\Controllers\Agent\EmailController::class, 'bulkDelete'])->name('agent.email.bulk-delete');
        Route::delete('/email/thread/{thread}', [App\Http\Controllers\Agent\EmailController::class, 'destroyThread'])->name('agent.email.destroy');
    });
});
