@extends('layouts.app')

@section('content')
<div class="min-h-screen bg-gray-50 py-12">
    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Email Signatures</h1>
                    <p class="mt-2 text-gray-600">Manage email signatures for {{ $mailAccount->email }}</p>
                </div>
                <div class="flex space-x-3">
                    <a href="{{ route('email.accounts') }}" 
                       class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path>
                        </svg>
                        Back to Accounts
                    </a>
                    <button onclick="showCreateSignatureModal()" 
                            class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                        </svg>
                        Create Signature
                    </button>
                </div>
            </div>
        </div>

        <!-- Signatures List -->
        <div class="bg-white shadow rounded-lg">
            <div class="px-6 py-4 border-b border-gray-200">
                <h3 class="text-lg font-medium text-gray-900">Existing Signatures</h3>
            </div>
            <div class="px-6 py-4">
                @if($signatures->count() > 0)
                    <div class="space-y-4">
                        @foreach($signatures as $signature)
                            <div class="border border-gray-200 rounded-lg p-4">
                                <div class="flex items-center justify-between">
                                    <div class="flex-1">
                                        <div class="flex items-center space-x-3">
                                            <h4 class="text-lg font-medium text-gray-900">{{ $signature->signature_name }}</h4>
                                            @if($signature->is_default)
                                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                                    Default
                                                </span>
                                            @endif
                                            @if(!$signature->is_active)
                                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                                                    Inactive
                                                </span>
                                            @endif
                                        </div>
                                        <div class="mt-2 text-sm text-gray-600">
                                            {{ Str::limit(strip_tags($signature->signature_html), 100) }}
                                        </div>
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <button onclick="previewSignature('{{ $signature->id }}')" 
                                                class="text-indigo-600 hover:text-indigo-900 text-sm font-medium">
                                            Preview
                                        </button>
                                        <button onclick="editSignature('{{ $signature->id }}')" 
                                                class="text-indigo-600 hover:text-indigo-900 text-sm font-medium">
                                            Edit
                                        </button>
                                        <button onclick="deleteSignature('{{ $signature->id }}')" 
                                                class="text-red-600 hover:text-red-900 text-sm font-medium">
                                            Delete
                                        </button>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="text-center py-8">
                        <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15.232 5.232l3.536 3.536m-2.036-5.036a2.5 2.5 0 113.536 3.536L6.5 21.036H3v-3.572L16.732 3.732z"></path>
                        </svg>
                        <h3 class="mt-2 text-sm font-medium text-gray-900">No signatures</h3>
                        <p class="mt-1 text-sm text-gray-500">Get started by creating your first email signature.</p>
                        <div class="mt-6">
                            <button onclick="showCreateSignatureModal()" 
                                    class="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700">
                                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                                </svg>
                                Create Signature
                            </button>
                        </div>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Create/Edit Signature Modal -->
<div id="signatureModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg p-6 max-w-2xl w-full mx-4 max-h-[90vh] overflow-y-auto">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-semibold" id="modalTitle">Create Signature</h3>
            <button onclick="hideSignatureModal()" class="text-gray-400 hover:text-gray-600">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        
        <form id="signatureForm" onsubmit="submitSignature(event)">
            <input type="hidden" id="signatureId" name="signature_id">
            <input type="hidden" name="mail_account_id" value="{{ $mailAccount->id }}">
            
            <div class="space-y-4">
                <div>
                    <label for="signatureName" class="block text-sm font-medium text-gray-700">Signature Name</label>
                    <input type="text" id="signatureName" name="signature_name" required
                           class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                </div>
                
                <div>
                    <label for="signatureHtml" class="block text-sm font-medium text-gray-700">Signature Content (HTML)</label>
                    <textarea id="signatureHtml" name="signature_html" rows="6" required
                              class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"
                              placeholder="Enter your signature HTML content..."></textarea>
                    <p class="mt-1 text-sm text-gray-500">You can use HTML tags for formatting (e.g., &lt;strong&gt;, &lt;em&gt;, &lt;br&gt;)</p>
                </div>
                
                <div>
                    <label for="signatureText" class="block text-sm font-medium text-gray-700">Plain Text Version</label>
                    <textarea id="signatureText" name="signature_text" rows="3" required
                              class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"
                              placeholder="Enter plain text version of your signature..."></textarea>
                </div>
                
                <div class="flex items-center">
                    <input type="checkbox" id="isDefault" name="is_default" value="1"
                           class="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300 rounded">
                    <label for="isDefault" class="ml-2 block text-sm text-gray-900">
                        Set as default signature
                    </label>
                </div>
            </div>
            
            <div class="mt-6 flex justify-end space-x-3">
                <button type="button" onclick="hideSignatureModal()" 
                        class="px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                    Cancel
                </button>
                <button type="submit" id="submitButton"
                        class="px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700">
                    Create Signature
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Preview Modal -->
<div id="previewModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg p-6 max-w-lg w-full mx-4">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-semibold">Signature Preview</h3>
            <button onclick="hidePreviewModal()" class="text-gray-400 hover:text-gray-600">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        <div id="previewContent" class="border border-gray-200 rounded-md p-4 bg-gray-50">
            <!-- Preview content will be loaded here -->
        </div>
    </div>
</div>

<script>
let currentSignatures = @json($signatures);

function showCreateSignatureModal() {
    document.getElementById('modalTitle').textContent = 'Create Signature';
    document.getElementById('signatureForm').reset();
    document.getElementById('signatureId').value = '';
    document.getElementById('submitButton').textContent = 'Create Signature';
    document.getElementById('signatureModal').classList.remove('hidden');
}

function editSignature(signatureId) {
    const signature = currentSignatures.find(s => s.id === signatureId);
    if (!signature) return;
    
    document.getElementById('modalTitle').textContent = 'Edit Signature';
    document.getElementById('signatureId').value = signatureId;
    document.getElementById('signatureName').value = signature.signature_name;
    document.getElementById('signatureHtml').value = signature.signature_html;
    document.getElementById('signatureText').value = signature.signature_text;
    document.getElementById('isDefault').checked = signature.is_default;
    document.getElementById('submitButton').textContent = 'Update Signature';
    document.getElementById('signatureModal').classList.remove('hidden');
}

function hideSignatureModal() {
    document.getElementById('signatureModal').classList.add('hidden');
}

function hidePreviewModal() {
    document.getElementById('previewModal').classList.add('hidden');
}

function previewSignature(signatureId) {
    const signature = currentSignatures.find(s => s.id === signatureId);
    if (!signature) return;
    
    document.getElementById('previewContent').innerHTML = signature.signature_html;
    document.getElementById('previewModal').classList.remove('hidden');
}

function submitSignature(event) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    const signatureId = formData.get('signature_id');
    const isEdit = !!signatureId;
    
    const url = isEdit 
        ? `/email-accounts/{{ $mailAccount->id }}/signatures/${signatureId}`
        : `/email-accounts/{{ $mailAccount->id }}/signatures`;
    
    const method = isEdit ? 'PUT' : 'POST';
    
    fetch(url, {
        method: method,
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({
            signature_name: formData.get('signature_name'),
            signature_html: formData.get('signature_html'),
            signature_text: formData.get('signature_text'),
            is_default: formData.get('is_default') === '1'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload(); // Refresh to show updated signatures
        } else {
            alert('Error: ' + data.error);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred while saving the signature');
    });
}

function deleteSignature(signatureId) {
    if (!confirm('Are you sure you want to delete this signature?')) {
        return;
    }
    
    fetch(`/email-accounts/{{ $mailAccount->id }}/signatures/${signatureId}`, {
        method: 'DELETE',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('Error: ' + data.error);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred while deleting the signature');
    });
}
</script>
@endsection
