/**
 * CSRF Token Setup for AJAX Requests
 * Ensures all AJAX requests include CSRF token for security
 */

// Get CSRF token from meta tag
const token = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');

if (token) {
    // Configure axios defaults
    if (window.axios) {
        window.axios.defaults.headers.common['X-CSRF-TOKEN'] = token;
    }
    
    // Configure fetch defaults
    window.csrfToken = token;
    
    // Override fetch to automatically include CSRF token
    const originalFetch = window.fetch;
    window.fetch = function(url, options = {}) {
        // Only add CSRF token for same-origin requests
        if (url.startsWith('/') || url.startsWith(window.location.origin)) {
            options.headers = {
                ...options.headers,
                'X-CSRF-TOKEN': token
            };
        }
        return originalFetch(url, options);
    };
    
    // Override XMLHttpRequest to automatically include CSRF token
    const originalXHROpen = XMLHttpRequest.prototype.open;
    XMLHttpRequest.prototype.open = function(method, url, ...args) {
        this._url = url;
        return originalXHROpen.apply(this, [method, url, ...args]);
    };
    
    const originalXHRSend = XMLHttpRequest.prototype.send;
    XMLHttpRequest.prototype.send = function(data) {
        // Only add CSRF token for same-origin requests
        if (this._url && (this._url.startsWith('/') || this._url.startsWith(window.location.origin))) {
            this.setRequestHeader('X-CSRF-TOKEN', token);
        }
        return originalXHRSend.apply(this, [data]);
    };
    
} else {
    console.warn('⚠️ CSRF token not found in meta tag');
}

// Utility function to get CSRF token
window.getCsrfToken = function() {
    return token;
};

// Utility function to add CSRF token to form data
window.addCsrfToFormData = function(formData) {
    if (token) {
        formData.append('_token', token);
    }
    return formData;
};

// Utility function to add CSRF token to object
window.addCsrfToObject = function(obj) {
    if (token) {
        obj._token = token;
    }
    return obj;
};

