<?php

namespace App\Traits;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

trait BelongsToUser
{
    /**
     * Boot the trait and add global scopes
     */
    protected static function bootBelongsToUser()
    {
        // Global scope for tenant isolation
        static::addGlobalScope('tenant', function (Builder $builder) {
            $user = Auth::user();

            if (!$user) {
                return;
            }

            if ($user->isSuperAdmin()) {
                // Super admin can see all records
                return;
            }

            // All non-super-admin users are limited to their tenant
            if ($user->tenant_id) {
                $builder->where('tenant_id', $user->tenant_id);
            }
        });

        // Global scope for user-level isolation
        static::addGlobalScope('user', function (Builder $builder) {
            $user = Auth::user();

            if (!$user) {
                return;
            }

            if ($user->isSuperAdmin()) {
                // Super admin can see all records
                return;
            }

            if ($user->isTenantAdmin()) {
                // Tenant admin can see all records in their tenant (handled by tenant scope)
                return;
            }

            // Agents only see their own records
            $builder->where('user_id', $user->id);
        });

        // Automatically set user_id and tenant_id when creating new records
        static::creating(function ($model) {
            $user = Auth::user();

            if ($user) {
                if (!$model->user_id) {
                    $model->user_id = $user->id;
                }
                if (!$model->tenant_id && $user->tenant_id) {
                    $model->tenant_id = $user->tenant_id;
                }
            }
        });
    }

    /**
     * Scope a query to only include records for the current user
     */
    public function scopeForUser(Builder $query, $userId = null)
    {
        $userId = $userId ?? Auth::id();
        return $query->where('user_id', $userId);
    }

    /**
     * Scope a query to only include records for the current tenant
     */
    public function scopeForTenant(Builder $query, $tenantId = null)
    {
        $user = Auth::user();
        
        if (!$user) {
            return $query;
        }

        if ($user->isSuperAdmin()) {
            // Super admin can see all tenants if tenant_id is specified
            if ($tenantId) {
                return $query->where('tenant_id', $tenantId);
            }
            return $query;
        }

        // Regular users only see their tenant's records
        return $query->where('tenant_id', $user->tenant_id);
    }

    /**
     * Scope a query to only include records for the current user and tenant
     */
    public function scopeForCurrentUser(Builder $query)
    {
        $user = Auth::user();
        
        if (!$user) {
            return $query;
        }

        if ($user->isSuperAdmin()) {
            return $query;
        }

        return $query->where('user_id', $user->id)
                    ->where('tenant_id', $user->tenant_id);
    }
}
