<?php

namespace App\Support;

use GuzzleHttp\Client;

/**
 * Helper class to create secure HTTP clients with proper SSL configuration
 *
 * SSL verification is only disabled in local development environment.
 * In production, staging, and testing environments, SSL is always verified.
 */
class SecureHttpClient
{
    /**
     * Check if SSL verification should be disabled
     * Only returns true in local development environment
     */
    public static function shouldDisableSslVerification(): bool
    {
        if (config('app.env') !== 'local' || config('app.debug') !== true) {
            return false;
        }

        $envOverride = env('AI_HTTP_DISABLE_SSL_VERIFY');
        if ($envOverride === null) {
            return true;
        }

        return filter_var($envOverride, FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Get SSL verification setting for Guzzle
     * Returns false only in local development, otherwise true
     */
    public static function getSslVerification(): bool
    {
        return !self::shouldDisableSslVerification();
    }

    /**
     * Create a Guzzle HTTP client with proper SSL configuration
     *
     * @param array $options Additional Guzzle options
     * @return Client
     */
    public static function create(array $options = []): Client
    {
        $defaultOptions = [
            'timeout' => 30,
            'verify' => self::getSslVerification(),
        ];

        // Only add CURL SSL options in local development
        if (self::shouldDisableSslVerification()) {
            $defaultOptions['curl'] = [
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => false,
            ];
        }

        return new Client(array_merge($defaultOptions, $options));
    }

    /**
     * Get HTTP client options for use with Laravel HTTP facade
     *
     * @return array
     */
    public static function getHttpOptions(): array
    {
        return [
            'verify' => self::getSslVerification(),
        ];
    }

    /**
     * Get stream context options for PHP stream functions
     * Used for SMTP/IMAP connections
     *
     * @return array
     */
    public static function getStreamContextOptions(): array
    {
        if (self::shouldDisableSslVerification()) {
            return [
                'ssl' => [
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                ],
            ];
        }

        return [
            'ssl' => [
                'verify_peer' => true,
                'verify_peer_name' => true,
                'allow_self_signed' => false,
            ],
        ];
    }
}
