<?php

namespace App\Support;

use App\Models\MailAccount;
use App\Models\OauthConnection;
use App\Models\User;
use Google\Client;
use Google\Service\Gmail;
use Google\Service\MyBusinessAccountManagement;
use Illuminate\Support\Facades\Log;
use App\Support\SecureHttpClient;

class GoogleClientFactory
{
    private ?Client $client = null;
    private ?OauthConnection $connection = null;

    public function __construct()
    {
        // Note: Connection will be set via setUser() or when getClient() is called
        // This allows factory to work in both web (authenticated) and queue (background) contexts
    }

    /**
     * Set the user for this client factory
     *
     * @param int|string $userId
     * @return self
     */
    public function setUser($userId): self
    {
        $user = User::find($userId);

        $query = OauthConnection::where('provider', 'google')
            ->where('user_id', $userId);

        if ($user?->tenant_id) {
            $query->where('tenant_id', $user->tenant_id);
        }

        $this->connection = $query->orderBy('updated_at', 'desc')->first();

        return $this;
    }

    /**
     * Get Google Client instance
     *
     * @param int|string|null $userId Optional user ID to get client for specific user
     */
    public function getClient($userId = null): Client
    {
        // If userId provided, get connection for that user
        if ($userId && !$this->connection) {
            $this->setUser($userId);
        }

        // If no connection yet, try to get for currently authenticated user (web context)
        if (!$this->connection && auth()->check()) {
            $this->setUser(auth()->id());
        }

        if (!$this->client) {
            $this->client = new Client();
            $this->configureClient();
        }

        return $this->client;
    }

    /**
     * Configure the Google Client
     */
    private function configureClient(): void
    {
        if (!$this->connection) {
            throw new \Exception('No Google OAuth connection found. Please connect your Google account first.');
        }

        $config = config('services.google');
        
        $this->client->setClientId($config['client_id']);
        $this->client->setClientSecret($config['client_secret']);
        $this->client->setRedirectUri($config['redirect']);
        
        // Set scopes
        $scopes = explode(',', $config['scopes']);
        $this->client->setScopes($scopes);
        
        // Configure HTTP client with environment-aware SSL settings
        $this->client->setHttpClient(SecureHttpClient::create());

        // Set access token
        $tokens = $this->connection->tokens;
        $this->client->setAccessToken($tokens);

        // Check if token is expired and refresh if needed
        if ($this->client->isAccessTokenExpired()) {
            $this->refreshToken();
        }
    }

    /**
     * Refresh the access token
     */
    private function refreshToken(): void
    {
        try {
            $refreshToken = $this->connection->tokens['refresh_token'] ?? null;
            
            if (!$refreshToken) {
                throw new \Exception('No refresh token available. Please re-authenticate.');
            }

            $this->client->refreshToken($refreshToken);
            $newTokens = $this->client->getAccessToken();
            
            // Update tokens in database
            $this->connection->update(['tokens' => $newTokens]);
            
            Log::info('Google access token refreshed successfully');
        } catch (\Exception $e) {
            Log::error('Failed to refresh Google access token: ' . $e->getMessage());
            throw new \Exception('Failed to refresh access token. Please re-authenticate.');
        }
    }

    /**
     * Configure the Google Client for a specific account
     */
    private function configureClientForAccount(Client $client, MailAccount $mailAccount): void
    {
        Log::info('GoogleClientFactory: Configuring client for account', [
            'account_email' => $mailAccount->email
        ]);
        
        $connection = $mailAccount->oauthConnection;
        
        if (!$connection) {
            Log::error('GoogleClientFactory: No OAuth connection found for mail account', [
                'account_email' => $mailAccount->email,
                'oauth_connection_id' => $mailAccount->oauth_connection_id
            ]);
            throw new \Exception('No OAuth connection found for this mail account.');
        }
        
        Log::info('GoogleClientFactory: Found OAuth connection', [
            'connection_id' => $connection->id,
            'account_email' => $connection->account_email
        ]);

        $config = config('services.google');
        
        $client->setClientId($config['client_id']);
        $client->setClientSecret($config['client_secret']);
        $client->setRedirectUri($config['redirect']);
        
        // Set scopes
        $scopes = explode(',', $config['scopes']);
        $client->setScopes($scopes);
        
        // Configure HTTP client with environment-aware SSL settings
        $client->setHttpClient(SecureHttpClient::create());
        
        // Set access token
        $tokens = $connection->tokens;
        $client->setAccessToken($tokens);

        // Check if token is expired and refresh if needed
        if ($client->isAccessTokenExpired()) {
            $this->refreshTokenForAccount($client, $connection);
        }
    }

    /**
     * Refresh the access token for a specific account
     */
    private function refreshTokenForAccount(Client $client, OauthConnection $connection): void
    {
        try {
            $refreshToken = $connection->tokens['refresh_token'] ?? null;
            
            if (!$refreshToken) {
                throw new \Exception('No refresh token available. Please re-authenticate.');
            }

            $client->refreshToken($refreshToken);
            $newTokens = $client->getAccessToken();
            
            // Update tokens in database
            $connection->update(['tokens' => $newTokens]);
            
            Log::info('Google access token refreshed successfully for account: ' . $connection->account_email);
        } catch (\Exception $e) {
            Log::error('Failed to refresh Google access token for account ' . $connection->account_email . ': ' . $e->getMessage());
            throw new \Exception('Failed to refresh access token. Please re-authenticate.');
        }
    }

    /**
     * Get Gmail service
     */
    public function getGmailService(): Gmail
    {
        return new Gmail($this->getClient());
    }

    /**
     * Get Gmail service for a specific mail account
     */
    public function getGmailServiceForAccount(MailAccount $mailAccount): Gmail
    {
        Log::info('GoogleClientFactory: Creating Gmail service for account', [
            'account_email' => $mailAccount->email,
            'oauth_connection_id' => $mailAccount->oauth_connection_id
        ]);
        
        $client = new Client();
        $this->configureClientForAccount($client, $mailAccount);
        
        Log::info('GoogleClientFactory: Gmail service created successfully');
        
        return new Gmail($client);
    }

    /**
     * Get My Business service (Google Business Profile)
     */
    public function getMyBusinessService(): MyBusinessAccountManagement
    {
        return new MyBusinessAccountManagement($this->getClient());
    }

    /**
     * Check if OAuth connection exists
     */
    public function hasConnection(): bool
    {
        return $this->connection !== null;
    }

    /**
     * Get the OAuth connection
     */
    public function getConnection(): ?OauthConnection
    {
        return $this->connection;
    }
}
