<?php

namespace App\Support;

/**
 * Base64 URL-safe encoding/decoding helper
 *
 * Gmail API requires base64url encoding (RFC 4648) instead of standard base64.
 * This class provides consistent encoding/decoding across the application.
 */
class Base64Url
{
    /**
     * Encode data using URL-safe base64 (RFC 4648)
     *
     * Converts standard base64 characters:
     * - '+' becomes '-'
     * - '/' becomes '_'
     * - Removes padding '=' characters
     *
     * @param string $data The data to encode
     * @return string URL-safe base64 encoded string
     */
    public static function encode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    /**
     * Decode URL-safe base64 encoded data (RFC 4648)
     *
     * Converts URL-safe base64 characters back to standard:
     * - '-' becomes '+'
     * - '_' becomes '/'
     * - Adds padding '=' characters as needed
     *
     * @param string $data The URL-safe base64 encoded data
     * @return string Decoded data
     */
    public static function decode(string $data): string
    {
        $remainder = strlen($data) % 4;
        if ($remainder) {
            $data .= str_repeat('=', 4 - $remainder);
        }
        return base64_decode(strtr($data, '-_', '+/'));
    }
}
