<?php

namespace App\Services;

use App\Models\MailAccount;

class SimpleIMAPTest
{
    /**
     * Test IMAP connection using basic socket connection (no PHP IMAP extension warnings)
     */
    public function testConnection(MailAccount $mailAccount): array
    {
        $config = $mailAccount->imap_config;
        
        if (empty($config['host']) || empty($config['port']) || 
            empty($config['username']) || empty($config['password'])) {
            return [
                'success' => false,
                'message' => '❌ Incomplete IMAP configuration. Missing host, port, username, or password.'
            ];
        }

        try {
            // Test basic connectivity first
            $socket = @fsockopen($config['host'], $config['port'], $errno, $errstr, 10);
            
            if (!$socket) {
                return [
                    'success' => false,
                    'message' => "❌ Cannot connect to {$config['host']}:{$config['port']} - {$errstr}"
                ];
            }
            
            fclose($socket);
            
            // If we can connect via socket, consider it "basic connectivity OK"
            // For detailed IMAP testing, we'd need different approach
            return [
                'success' => true,
                'message' => "✅ Basic connectivity to {$config['host']}:{$config['port']} successful!\n\n" .
                           "📡 Server is reachable\n" .
                           "🔧 Note: This tests basic connectivity only.\n" .
                           "💡 For full IMAP validation, check your email client settings.",
                'recommended_folder' => 'INBOX'
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => '❌ IMAP connectivity test failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Check if IMAP extension is available
     */
    public function checkIMAPExtension(): array
    {
        if (!extension_loaded('imap')) {
            return [
                'success' => false,
                'available' => false,
                'message' => '❌ PHP IMAP extension is not installed. This server setup cannot do full IMAP operations.'
            ];
        }

        return [
            'success' => true,
            'available' => true,
            'message' => '✅ PHP IMAP extension is available.'
        ];
    }

    /**
     * Get basic connection info
     */
    public function getConnectionInfo(array $config): array
    {
        return [
            'host' => $config['host'],
            'port' => $config['port'],
            'encryption' => $config['encryption'],
            'username' => $config['username'],
            'folder' => $config['folder'] ?? 'INBOX',
            'connection_string' => $this->buildConnectionString($config)
        ];
    }

    /**
     * Build connection string
     */
    private function buildConnectionString(array $config): string
    {
        $flags = [];
        
        if ($config['encryption'] === 'ssl') {
            $flags[] = 'ssl';
        } elseif ($config['encryption'] === 'tls') {
            $flags[] = 'tls';
        }
        
        $flags[] = 'novalidate-cert';
        
        $flagsString = !empty($flags) ? '/' . implode('/', $flags) : '';
        
        return "{" . $config['host'] . ":" . $config['port'] . $flagsString . "}";
    }

    /**
     * Auto-resolve IMAP host based on email domain
     */
    public function resolveImapHost(string $email): string
    {
        $domain = strtolower(substr(strrchr($email, "@"), 1));
        
        $imapHosts = [
            'gmail.com' => 'imap.gmail.com',
            'googlemail.com' => 'imap.gmail.com',
            'outlook.com' => 'outlook.office365.com',
            'hotmail.com' => 'outlook.office365.com',
            'live.com' => 'outlook.office365.com',
            'yahoo.com' => 'imap.mail.yahoo.com',
            'ymail.com' => 'imap.mail.yahoo.com',
            'aol.com' => 'imap.aol.com',
        ];
        
        $providerPatterns = [
            'hostinger' => 'imap.hostinger.com',
            'godaddy' => 'imap.secureserver.net',
            'bluehost' => 'imap.bluehost.com',
            'siteground' => 'imap.siteground.com',
            'namecheap' => 'mail.privateemail.com',
            'name.com' => 'imap.name.com',
            'titan.email' => 'imap.titan.email',
        ];
        
        if (isset($imapHosts[$domain])) {
            return $imapHosts[$domain];
        }
        
        foreach ($providerPatterns as $provider => $host) {
            if (strpos($domain, $provider) !== false) {
                return $host;
            }
        }
        
        return "mail.{$domain}";
    }
}
