<?php

namespace App\Services;

use App\Support\GoogleClientFactory;
use Illuminate\Support\Facades\Log;

class GoogleBusinessProfile
{
    private GoogleClientFactory $clientFactory;

    public function __construct(GoogleClientFactory $clientFactory)
    {
        $this->clientFactory = $clientFactory;
    }

    /**
     * List all locations for the authenticated account
     * 
     * @return array
     */
    public function listLocations(): array
    {
        try {
            $client = $this->clientFactory->getClient();
            
            Log::info('Fetching locations from Google Business Profile API');
            
            // Use the My Business Account Management API endpoint
            $httpClient = $client->getHttpClient();
            $url = "https://mybusinessaccountmanagement.googleapis.com/v1/accounts";
            
            $response = $httpClient->get($url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $client->getAccessToken()['access_token'],
                    'Content-Type' => 'application/json',
                ]
            ]);
            
            $data = json_decode($response->getBody()->getContents(), true);
            
            if (!isset($data['accounts'])) {
                Log::warning('No accounts found in API response', ['response' => $data]);
                return [];
            }
            
            $allLocations = [];
            
            foreach ($data['accounts'] as $account) {
                $accountName = $account['name'];
                Log::info('Processing account', ['account_name' => $accountName]);
                
                // Get locations for this account
                $accountLocations = $this->getLocationsForAccount($client, $accountName);
                $allLocations = array_merge($allLocations, $accountLocations);
            }
            
            // If no locations found, return empty array (DO NOT return test data)
            if (empty($allLocations)) {
                Log::warning('No locations found. Check OAuth permissions, API access, and that the account has locations.');
                return [];
            }
            
            Log::info('Successfully fetched locations from Google Business Profile', [
                'location_count' => count($allLocations)
            ]);
            
            return $allLocations;
            
        } catch (\Exception $e) {
            Log::error('Failed to list Google Business Profile locations: ' . $e->getMessage(), [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Return empty array if API fails (DO NOT return test data)
            Log::error('Google Business Profile API failed: ' . $e->getMessage());
            Log::error('Action Required: 1) Check OAuth connection 2) Verify API is enabled 3) Check scopes include business.manage');
            return [];
        }
    }

    /**
     * Get locations for a specific account
     * 
     * @param \Google\Client $client
     * @param string $accountName
     * @return array
     */
    private function getLocationsForAccount($client, string $accountName): array
    {
        try {
            Log::info('Fetching locations for account', ['account_name' => $accountName]);
            
            // Use the full resource name as-is (e.g. "accounts/12345678901234567890")
            // Do NOT extract just the numeric ID - use the full resource name
            $httpClient = $client->getHttpClient();
            $url = "https://mybusinessbusinessinformation.googleapis.com/v1/{$accountName}/locations";
            
            // Add readMask parameter as required by the API
            $url .= "?readMask=name,title,websiteUri,storeCode,metadata";
            
            Log::info('Making API request to locations endpoint', [
                'account_name' => $accountName,
                'url' => $url
            ]);
            
            $response = $httpClient->get($url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $client->getAccessToken()['access_token'],
                    'Content-Type' => 'application/json',
                ]
            ]);
            
            $data = json_decode($response->getBody()->getContents(), true);
            
            if (!isset($data['locations'])) {
                Log::warning('No locations found for account', [
                    'account_name' => $accountName,
                    'response' => $data
                ]);
                return [];
            }
            
            $result = [];
            foreach ($data['locations'] as $location) {
                $result[] = [
                    'account_id' => $accountName, // Keep the full resource name
                    'location_id' => $location['name'],
                    'display_name' => $location['title'] ?? $location['name'],
                    'address' => $location['address']['formattedAddress'] ?? null,
                    'phone' => $location['primaryPhone'] ?? null,
                    'website' => $location['websiteUri'] ?? null, // Note: websiteUri not websiteUrl
                ];
            }
            
            Log::info('Successfully fetched locations for account', [
                'account_name' => $accountName,
                'location_count' => count($result)
            ]);
            
            return $result;
            
        } catch (\Exception $e) {
            Log::error('Failed to fetch locations for account', [
                'account_name' => $accountName,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return [];
        }
    }

    /**
     * List reviews for a specific location
     *
     * @param string $accountId
     * @param string $locationId
     * @param int|string|null $userId User ID to get OAuth token for (optional, for queue context)
     * @return array
     */
    public function listReviews(string $accountId, string $locationId, $userId = null): array
    {
        try {
            $client = $this->clientFactory->getClient($userId);
            
            // Use proper resource name format: accounts/{accountId}/locations/{locationId}
            $name = "accounts/{$accountId}/locations/{$locationId}";
            
            Log::info('Fetching reviews from Google Business Profile API', [
                'account_id' => $accountId,
                'location_id' => $locationId,
                'resource_name' => $name
            ]);
            
            // Use the My Business API v4 endpoint for reviews
            $httpClient = $client->getHttpClient();
            $url = "https://mybusiness.googleapis.com/v4/{$name}/reviews";
            
            $response = $httpClient->get($url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $client->getAccessToken()['access_token'],
                    'Content-Type' => 'application/json',
                ]
            ]);
            
            $data = json_decode($response->getBody()->getContents(), true);
            
            if (!isset($data['reviews'])) {
                Log::warning('No reviews found in API response', [
                    'account_id' => $accountId,
                    'location_id' => $locationId,
                    'response' => $data
                ]);
                return [];
            }
            
            $reviews = [];
            foreach ($data['reviews'] as $review) {
                $reviews[] = [
                    'id' => $review['reviewId'] ?? uniqid('review_'),
                    'comment' => $review['comment'] ?? '',
                    'stars' => $review['starRating'] ?? 0,
                    'lang' => $review['languageCode'] ?? 'en',
                    'createTime' => $review['createTime'] ?? now()->toISOString(),
                    'reviewer' => [
                        'displayName' => $review['reviewer']['displayName'] ?? 'Anonymous',
                        'profilePhotoUrl' => $review['reviewer']['profilePhotoUrl'] ?? null,
                        'isAnonymous' => $review['reviewer']['isAnonymous'] ?? true,
                    ],
                    'reviewUrl' => $review['reviewUrl'] ?? null,
                ];
            }
            
            Log::info('Successfully fetched reviews from Google Business Profile', [
                'account_id' => $accountId,
                'location_id' => $locationId,
                'review_count' => count($reviews)
            ]);
            
            return $reviews;
            
        } catch (\Exception $e) {
            Log::error('Failed to fetch reviews from Google Business Profile API', [
                'account_id' => $accountId,
                'location_id' => $locationId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Return empty array if API fails (DO NOT return test data)
            Log::error('Google Reviews API failed: ' . $e->getMessage());
            Log::error('Check if: 1) Account/Location IDs are test data 2) OAuth token is valid 3) API scopes are correct');
            return [];
        }
    }

    /**
     * Reply to a review
     * 
     * @param string $accountId
     * @param string $locationId
     * @param string $reviewId
     * @param string $text
     * @return mixed
     */
    public function replyToReview(string $accountId, string $locationId, string $reviewId, string $text)
    {
        try {
            $client = $this->clientFactory->getClient();
            
            // Use proper resource name format: accounts/{accountId}/locations/{locationId}/reviews/{reviewId}
            $name = "accounts/{$accountId}/locations/{$locationId}/reviews/{$reviewId}";
            
            Log::info('Posting reply to Google Business Profile review', [
                'account_id' => $accountId,
                'location_id' => $locationId,
                'review_id' => $reviewId,
                'resource_name' => $name,
                'reply_text' => $text
            ]);
            
            // Use the My Business API v4 endpoint for replies
            $httpClient = $client->getHttpClient();
            $url = "https://mybusiness.googleapis.com/v4/{$name}/reply";
            
            $response = $httpClient->put($url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $client->getAccessToken()['access_token'],
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'comment' => $text
                ]
            ]);
            
            $data = json_decode($response->getBody()->getContents(), true);
            
            Log::info('Successfully posted reply to Google Business Profile review', [
                'account_id' => $accountId,
                'location_id' => $locationId,
                'review_id' => $reviewId,
                'response' => $data
            ]);
            
            return [
                'success' => true,
                'message' => 'Reply posted successfully',
                'data' => $data
            ];
            
        } catch (\Exception $e) {
            Log::error('Failed to post reply to Google Business Profile review', [
                'account_id' => $accountId,
                'location_id' => $locationId,
                'review_id' => $reviewId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return [
                'success' => false,
                'message' => 'Failed to post reply: ' . $e->getMessage(),
            ];
        }
    }
}
