<?php

namespace App\Services;

use App\Models\MailAccount;
use App\Support\GoogleClientFactory;
use App\Support\Base64Url;
use Google\Service\Gmail\Message;
use Google\Service\Gmail\MessagePart;
use Illuminate\Support\Facades\Log;

class GmailService
{
    private GoogleClientFactory $clientFactory;

    public function __construct(GoogleClientFactory $clientFactory)
    {
        $this->clientFactory = $clientFactory;
    }

    /**
     * List recent message IDs from Gmail
     * 
     * @param MailAccount $mailAccount
     * @param string $query
     * @param int $maxResults
     * @return array
     */
    public function listRecentMessageIds(MailAccount $mailAccount, string $query = 'newer_than:1d -from:me in:inbox', int $maxResults = 25): array
    {
        try {
            Log::info('GmailService: Starting listRecentMessageIds', [
                'account_email' => $mailAccount->email,
                'query' => $query,
                'maxResults' => $maxResults
            ]);
            
            $service = $this->clientFactory->getGmailServiceForAccount($mailAccount);
            
            Log::info('GmailService: Got Gmail service, making API call');
            
            $response = $service->users_messages->listUsersMessages('me', [
                'q' => $query,
                'maxResults' => $maxResults,
            ]);
            
            $messageIds = [];
            foreach ($response->getMessages() as $message) {
                $messageIds[] = $message->getId();
            }
            
            Log::info('GmailService: Retrieved message IDs', [
                'count' => count($messageIds),
                'messageIds' => $messageIds
            ]);
            
            return $messageIds;
        } catch (\Exception $e) {
            Log::error('GmailService: Failed to list Gmail message IDs', [
                'error' => $e->getMessage(),
                'account_email' => $mailAccount->email,
                'trace' => $e->getTraceAsString()
            ]);
            return [];
        }
    }

    /**
     * Get full message details
     * 
     * @param MailAccount $mailAccount
     * @param string $messageId
     * @return array
     */
    public function getMessageFull(MailAccount $mailAccount, string $messageId): array
    {
        try {
            $service = $this->clientFactory->getGmailServiceForAccount($mailAccount);
            $message = $service->users_messages->get('me', $messageId, ['format' => 'full']);
            
            $headers = $this->extractHeaders($message);
            $body = $this->extractBody($service, $message);
            
            return [
                'id' => $message->getId(),
                'threadId' => $message->getThreadId(),
                'headers' => $headers,
                'snippet' => $message->getSnippet(),
                'body_text' => $body['text'],
                'body_html' => $body['html'],
            ];
        } catch (\Exception $e) {
            Log::error('Failed to get Gmail message: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Extract headers from Gmail message
     * 
     * @param Message $message
     * @return array
     */
    private function extractHeaders(Message $message): array
    {
        $headers = [];
        
        foreach ($message->getPayload()->getHeaders() as $header) {
            $headers[$header->getName()] = $header->getValue();
        }
        
        return $headers;
    }

    /**
     * Extract body from Gmail message
     *
     * Uses Base64Url helper to decode Gmail's URL-safe base64 encoded content
     *
     * @param Message $message
     * @return array
     */
    private function extractBody($service, Message $message): array
    {
        $payload = $message->getPayload();
        $body = ['text' => '', 'html' => ''];

        $fetchAttachment = function (?string $attachmentId) use ($service, $message) {
            if (!$attachmentId) {
                return null;
            }

            try {
                $attachment = $service->users_messages_attachments->get('me', $message->getId(), $attachmentId);
                $data = $attachment->getData();
                return $data ? Base64Url::decode($data) : null;
            } catch (\Exception $e) {
                Log::warning('Failed to fetch Gmail attachment body', [
                    'message_id' => $message->getId(),
                    'attachment_id' => $attachmentId,
                    'error' => $e->getMessage(),
                ]);
                return null;
            }
        };

        $extractPart = function (MessagePart $part) use (&$extractPart, &$body, $fetchAttachment) {
            $mimeType = $part->getMimeType();
            $partBody = $part->getBody();
            $data = $partBody ? $partBody->getData() : null;
            $attachmentId = $partBody ? $partBody->getAttachmentId() : null;

            if ($mimeType === 'text/plain') {
                if ($data) {
                    $body['text'] = Base64Url::decode($data);
                } elseif ($attachmentId) {
                    $fetched = $fetchAttachment($attachmentId);
                    if ($fetched !== null) {
                        $body['text'] = $fetched;
                    }
                }
            } elseif ($mimeType === 'text/html') {
                if ($data) {
                    $body['html'] = Base64Url::decode($data);
                } elseif ($attachmentId) {
                    $fetched = $fetchAttachment($attachmentId);
                    if ($fetched !== null) {
                        $body['html'] = $fetched;
                    }
                }
            }

            $subParts = $part->getParts();
            if (!empty($subParts)) {
                foreach ($subParts as $subPart) {
                    $extractPart($subPart);
                }
            }
        };

        $extractPart($payload);

        return $body;
    }

    /**
     * Send a reply to a Gmail thread
     * 
     * @param MailAccount $mailAccount
     * @param string $threadId
     * @param string $to
     * @param string $subject
     * @param string $html
     * @param string $inReplyToMessageId
     * @return mixed
     */
    public function sendReply(MailAccount $mailAccount, string $threadId, string $to, string $subject, string $html, string $inReplyToMessageId)
    {
        try {
            $service = $this->clientFactory->getGmailServiceForAccount($mailAccount);
            
            // Create MIME message
            $message = $this->createMimeMessage($threadId, $to, $subject, $html, $inReplyToMessageId);
            
            // Send the message
            $result = $service->users_messages->send('me', $message);
            
            Log::info('Gmail reply sent successfully', ['messageId' => $result->getId()]);
            
            return [
                'success' => true,
                'messageId' => $result->getId(),
                'threadId' => $threadId,
            ];
        } catch (\Exception $e) {
            Log::error('Failed to send Gmail reply: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to send reply: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Create MIME message for Gmail
     *
     * Uses Base64Url helper to encode the message as required by Gmail API
     *
     * @param string $threadId
     * @param string $to
     * @param string $subject
     * @param string $html
     * @param string $inReplyToMessageId
     * @return Message
     */
    private function createMimeMessage(string $threadId, string $to, string $subject, string $html, string $inReplyToMessageId): Message
    {
        $boundary = uniqid(rand(), true);

        $rawMessage = "To: {$to}\r\n";
        $rawMessage .= "Subject: {$subject}\r\n";
        $rawMessage .= "In-Reply-To: {$inReplyToMessageId}\r\n";
        $rawMessage .= "References: {$inReplyToMessageId}\r\n";
        $rawMessage .= "MIME-Version: 1.0\r\n";
        $rawMessage .= "Content-Type: multipart/alternative; boundary=\"{$boundary}\"\r\n";
        $rawMessage .= "\r\n";

        // Add HTML part
        $rawMessage .= "--{$boundary}\r\n";
        $rawMessage .= "Content-Type: text/html; charset=UTF-8\r\n";
        $rawMessage .= "\r\n";
        $rawMessage .= $html . "\r\n";
        $rawMessage .= "--{$boundary}--\r\n";

        $message = new Message();
        $message->setRaw(Base64Url::encode($rawMessage));
        $message->setThreadId($threadId);

        return $message;
    }
}
