<?php

namespace App\Services;

use App\Models\Message;
use App\Models\Review;
use App\Models\Thread;
use App\Contracts\AIProviderInterface;
use Illuminate\Support\Facades\Log;

class EnhancedReplyDraftService
{
    private AIServiceManager $aiManager;
    private ?string $preferredProvider;

    public function __construct(AIServiceManager $aiManager, ?string $preferredProvider = null)
    {
        $this->aiManager = $aiManager;
        $this->preferredProvider = $preferredProvider;
    }

    /**
     * Generate AI draft for a review using the configured AI provider
     * 
     * @param Review $review
     * @param string|null $provider Override the default provider
     * @return array
     */
    public function makeForReview(Review $review, ?string $provider = null): array
    {
        $prompt = $this->buildReviewPrompt($review);
        
        try {
            $response = $this->generateWithFallback(
                'You are a helpful assistant that generates professional, warm, and concise replies to business reviews. Keep replies under 80 words and always thank the customer.',
                $prompt,
                ['max_tokens' => 900, 'temperature' => 0.7],
                $provider
            );
            
            if (!$response['success']) {
                throw new \Exception($response['error'] ?? 'Unknown AI error');
            }
            
            $draft = $response['content'];

            // Retry once with stronger guidance if we get an unusually short reply.
            if (str_word_count(strip_tags($draft)) < 20) {
                $expandedPrompt = $prompt
                    . "\n\nIMPORTANT: Your reply must be 2-4 sentences and at least 40 words. "
                    . "Thank the customer, reference their feedback, keep it warm and professional, "
                    . "and end with a complete sentence.";

                $retryResponse = $this->generateWithFallback(
                    'You are a helpful assistant that generates professional, warm, and concise replies to business reviews. Keep replies under 80 words and always thank the customer.',
                    $expandedPrompt,
                    ['max_tokens' => 200, 'temperature' => 0.7],
                    $provider
                );

                if (!empty($retryResponse['success'])) {
                    $draft = $retryResponse['content'] ?? $draft;
                }
            }
            
            // Enforce word limit
            $draft = $this->enforceWordLimit($draft, 80);
            
            // Check for escalation keywords
            $meta = $this->checkEscalationKeywords($review->text);
            $meta['ai_provider'] = $response['provider'];
            
            return [
                'draft' => $draft,
                'meta' => $meta,
            ];
        } catch (\Exception $e) {
            Log::error('Failed to generate review draft: ' . $e->getMessage());
            
            return [
                'draft' => '',
                'meta' => [
                    'error' => true, 
                    'message' => $this->summarizeError($e->getMessage()),
                    'ai_provider' => null
                ],
            ];
        }
    }

    /**
     * Generate AI draft for an email using the configured AI provider
     * 
     * @param Thread $thread
     * @param Message $lastInbound
     * @param string|null $provider Override the default provider
     * @return array
     */
    public function makeForEmail(Thread $thread, Message $lastInbound, ?string $provider = null): array
    {
        $prompt = $this->buildEmailPrompt($thread, $lastInbound);
        
        try {
            $response = $this->generateWithFallback(
                'You are a helpful assistant that generates professional, warm, and concise email replies. Keep replies under 120 words, be factual only, and include helpful next steps. Return only the email body (no subject line). Use 3-6 sentences.',
                $prompt,
                ['max_tokens' => 900, 'temperature' => 0.7],
                $provider
            );
            
            if (!$response['success']) {
                throw new \Exception($response['error'] ?? 'Unknown AI error');
            }
            
            $draft = $response['content'];

            // Retry once with stronger guidance if Gemini returns a very short reply.
            if (str_word_count(strip_tags($draft)) < 40) {
                $expandedPrompt = $prompt
                    . "\n\nIMPORTANT: Your reply must be 3-6 sentences and at least 80 words. "
                    . "Reference key details from the message and provide clear, helpful next steps.";

                $retryResponse = $this->generateWithFallback(
                    'You are a helpful assistant that generates professional, warm, and concise email replies. Keep replies under 120 words, be factual only, and include helpful next steps. Return only the email body (no subject line). Use 3-6 sentences.',
                    $expandedPrompt,
                    ['max_tokens' => 900, 'temperature' => 0.7],
                    $provider
                );

                if (!empty($retryResponse['success'])) {
                    $draft = $retryResponse['content'] ?? $draft;
                }
            }
            
            // Convert to HTML
            $draftHtml = $this->convertToHtml($draft);
            
            // Enforce word limit
            $draftHtml = $this->enforceWordLimit($draftHtml, 120);
            
            // Check for escalation keywords
            $meta = $this->checkEscalationKeywords($lastInbound->body_text);
            $meta['ai_provider'] = $response['provider'];
            
            return [
                'draft' => $draftHtml,
                'meta' => $meta,
            ];
        } catch (\Exception $e) {
            Log::error('Failed to generate email draft: ' . $e->getMessage());
            
            return [
                'draft' => '',
                'meta' => [
                    'error' => true, 
                    'message' => $this->summarizeError($e->getMessage()),
                    'ai_provider' => null
                ],
            ];
        }
    }

    /**
     * Get the AI provider to use
     * 
     * @param string|null $provider
     * @return AIProviderInterface
     */
    private function getAIProvider(?string $provider = null): AIProviderInterface
    {
        $providerToUse = $provider ?? $this->preferredProvider;
        
        if ($providerToUse) {
            try {
                return $this->aiManager->getProvider($providerToUse);
            } catch (\Exception $e) {
                Log::warning("Requested AI provider '{$providerToUse}' not available, falling back to default");
            }
        }
        
        return $this->aiManager->getDefaultProvider();
    }

    /**
     * Get available AI providers info
     * 
     * @return array
     */
    public function getAvailableProviders(): array
    {
        return $this->aiManager->getProviderInfo();
    }

    /**
     * Test all AI providers
     * 
     * @return array
     */
    public function testAllProviders(): array
    {
        return $this->aiManager->testAllProviders();
    }

    /**
     * Build prompt for review reply
     * 
     * @param Review $review
     * @return string
     */
    private function buildReviewPrompt(Review $review): string
    {
        $locationName = $review->location->display_name ?? 'our business';
        
        return "Generate a professional reply to this {$review->stars}-star review for {$locationName}:\n\n" .
               "Review: {$review->text}\n\n" .
               "Requirements:\n" .
               "- Thank the customer\n" .
               "- Address their concern if mentioned\n" .
               "- Keep it under 80 words\n" .
               "- Be warm and professional\n" .
               "- Don't make promises beyond policy";
    }

    /**
     * Build prompt for email reply
     * 
     * @param Thread $thread
     * @param Message $lastInbound
     * @return string
     */
    private function buildEmailPrompt(Thread $thread, Message $lastInbound): string
    {
        $plainText = trim((string) $lastInbound->body_text);
        $htmlText = trim((string) ($lastInbound->body_html ?? ''));
        $htmlStripped = $htmlText !== '' ? trim(html_entity_decode(strip_tags($htmlText), ENT_QUOTES | ENT_HTML5, 'UTF-8')) : '';

        // Prefer the richer body when plain text is too short.
        if ($plainText === '' && $htmlStripped !== '') {
            $messageText = $htmlStripped;
        } elseif ($plainText !== '' && $htmlStripped !== '' && strlen($plainText) < 200 && strlen($htmlStripped) > strlen($plainText)) {
            $messageText = $htmlStripped;
        } elseif ($plainText !== '') {
            $messageText = $plainText;
        } else {
            $messageText = $lastInbound->snippet ?? '';
        }

        $messageText = \Illuminate\Support\Str::limit(trim($messageText), 2000, '...');

        Log::info('Email draft prompt source', [
            'thread_id' => $thread->id ?? null,
            'message_id' => $lastInbound->id ?? null,
            'provider_msg_id' => $lastInbound->provider_msg_id ?? null,
            'body_text_length' => strlen($plainText),
            'body_html_length' => strlen($htmlText),
            'html_stripped_length' => strlen($htmlStripped),
            'snippet_length' => strlen((string) ($lastInbound->snippet ?? '')),
            'chosen_message_length' => strlen($messageText),
        ]);

        return "Generate a professional email reply to this message:\n\n" .
               "Subject: {$thread->subject}\n" .
               "From: {$lastInbound->from_addr}\n" .
               "Message: {$messageText}\n\n" .
               "Requirements:\n" .
               "- Be factual only\n" .
               "- Include helpful next step\n" .
               "- Keep it under 120 words\n" .
               "- Be warm and professional\n" .
               "- Return only the email body (no subject line)\n" .
               "- Use 3-6 sentences";
    }

    /**
     * Check for escalation keywords in text
     * 
     * @param string $text
     * @return array
     */
    private function checkEscalationKeywords(string $text): array
    {
        $keywords = ['injury', 'allergic', 'lawsuit', 'chargeback', 'refund'];
        $text = strtolower($text);
        
        foreach ($keywords as $keyword) {
            if (strpos($text, $keyword) !== false) {
                return ['escalate' => true, 'keyword' => $keyword];
            }
        }
        
        return ['escalate' => false];
    }

    /**
     * Enforce word limit on text
     * 
     * @param string $text
     * @param int $limit
     * @return string
     */
    private function enforceWordLimit(string $text, int $limit): string
    {
        $words = explode(' ', $text);
        
        if (count($words) <= $limit) {
            return $text;
        }
        
        return implode(' ', array_slice($words, 0, $limit)) . '...';
    }

    /**
     * Convert plain text to HTML
     * 
     * @param string $text
     * @return string
     */
    private function convertToHtml(string $text): string
    {
        // Simple conversion - wrap paragraphs in <p> tags
        $paragraphs = explode("\n\n", $text);
        $html = '';
        
        foreach ($paragraphs as $paragraph) {
            if (trim($paragraph)) {
                $html .= '<p>' . nl2br(htmlspecialchars(trim($paragraph))) . '</p>';
            }
        }
        
        return $html ?: '<p>' . htmlspecialchars($text) . '</p>';
    }

    /**
     * Generate a response using the preferred provider, falling back to others if needed
     *
     * @param string $systemPrompt
     * @param string $userPrompt
     * @param array $options
     * @param string|null $provider
     * @return array
     */
    private function generateWithFallback(string $systemPrompt, string $userPrompt, array $options, ?string $provider = null): array
    {
        $availableProviders = $this->aiManager->getAvailableProviders();
        $orderedProviders = $this->buildProviderOrder($provider, array_keys($availableProviders));
        $attempts = [];
        $lastError = null;

        foreach ($orderedProviders as $providerKey) {
            if (!isset($availableProviders[$providerKey])) {
                continue;
            }

            $providerInstance = $availableProviders[$providerKey];

            if (!$providerInstance->isConfigured()) {
                $attempts[] = "{$providerKey}: not configured";
                continue;
            }

            $response = $providerInstance->generateResponse($systemPrompt, $userPrompt, $options);

            if (!empty($response['success'])) {
                return [
                    'success' => true,
                    'content' => $response['content'] ?? '',
                    'provider' => $response['provider'] ?? $providerKey,
                    'attempts' => $attempts,
                ];
            }

            $lastError = $response['error'] ?? 'Unknown AI error';
            $attempts[] = "{$providerKey}: {$lastError}";
        }

        return [
            'success' => false,
            'error' => $this->summarizeError($lastError ?? 'No AI providers are available'),
            'attempts' => $attempts,
        ];
    }

    /**
     * Build provider order: requested provider, default provider, then others
     *
     * @param string|null $requested
     * @param array $availableKeys
     * @return array
     */
    private function buildProviderOrder(?string $requested, array $availableKeys): array
    {
        $defaultProvider = config('services.ai.default_provider', 'gemini');
        $ordered = [];

        if ($requested) {
            $ordered[] = $requested;
        }

        if ($defaultProvider) {
            $ordered[] = $defaultProvider;
        }

        foreach ($availableKeys as $key) {
            $ordered[] = $key;
        }

        return array_values(array_unique($ordered));
    }

    /**
     * Summarize provider errors for user-facing messages
     *
     * @param string $message
     * @return string
     */
    private function summarizeError(string $message): string
    {
        $lower = strtolower($message);

        if (str_contains($lower, 'quota') || str_contains($lower, 'resource_exhausted')) {
            return 'AI provider quota exceeded. Configure a different provider or upgrade your plan.';
        }

        if (str_contains($lower, 'api key') || str_contains($lower, 'not configured')) {
            return 'AI provider API key not configured.';
        }

        if (str_contains($lower, 'not available')) {
            return 'AI provider not available.';
        }

        return $message;
    }
}
