<?php

namespace App\Services;

use App\Models\MailAccount;
use App\Models\Thread;
use App\Models\Message;
use Illuminate\Support\Facades\Log;

/**
 * Email Ingestion Service
 *
 * Centralized service for ingesting emails from mail accounts.
 * Used by both EmailController (manual) and IngestEmailsJob (automatic).
 *
 * Reference: AUTO_INGESTION.md Section 4.4
 */
class EmailIngestionService
{
    private GmailService $gmailService;
    private IMAPEmailService $imapService;

    public function __construct(GmailService $gmailService, IMAPEmailService $imapService)
    {
        $this->gmailService = $gmailService;
        $this->imapService = $imapService;
    }

    /**
     * Ingest emails for a specific mail account
     *
     * @param MailAccount $mailAccount
     * @return array Result with success status and counts
     * @throws \Exception
     */
    public function ingestForAccount(MailAccount $mailAccount): array
    {
        Log::info('Starting email ingestion for account', [
            'email' => $mailAccount->email,
            'connection_type' => $mailAccount->connection_type
        ]);

        // Route to appropriate service based on account type
        if ($mailAccount->isOAuth()) {
            return $this->ingestOAuthEmails($mailAccount);
        } elseif ($mailAccount->hasImapConfig()) {
            return $this->ingestImapEmails($mailAccount);
        } else {
            throw new \Exception('Email account not configured for fetching. Please configure OAuth or IMAP settings.');
        }
    }

    /**
     * Fetch emails from OAuth account (Gmail)
     *
     * @param MailAccount $mailAccount
     * @return array
     * @throws \Exception
     */
    private function ingestOAuthEmails(MailAccount $mailAccount): array
    {
        try {
            $messageIds = $this->gmailService->listRecentMessageIds($mailAccount);
            $ingestedCount = 0;
            $threadsCreated = 0;
            $messagesCreated = 0;

            foreach ($messageIds as $messageId) {
                $messageData = $this->gmailService->getMessageFull($mailAccount, $messageId);

                if (empty($messageData)) {
                    continue;
                }

                $headers = $messageData['headers'];
                $threadId = $messageData['threadId'];

                // Create or update thread
                $thread = Thread::updateOrCreate(
                    ['provider_thread_id' => $threadId],
                    [
                        'mail_account_id' => $mailAccount->id,
                        'user_id' => $mailAccount->user_id,
                        'tenant_id' => $mailAccount->tenant_id,
                        'subject' => $headers['Subject'] ?? 'No Subject',
                        'participants' => $this->extractParticipants($headers),
                        'last_message_at' => now(),
                        'status' => 'open',
                    ]
                );

                if ($thread->wasRecentlyCreated) {
                    $threadsCreated++;
                }

                // Always update last_message_at to current time for fetched emails
                $thread->update(['last_message_at' => now()]);

                // Sanitize and process email body content
                $bodyHtml = $this->sanitizeUtf8($messageData['body_html'] ?? '');
                $bodyText = $this->sanitizeUtf8($messageData['body_text'] ?? '');

                // Check if body_text contains HTML (some email providers send HTML as plain text)
                $bodyTextHasHtml = !empty($bodyText) && (
                    stripos($bodyText, '<!DOCTYPE') !== false ||
                    stripos($bodyText, '<html') !== false ||
                    stripos($bodyText, '<body') !== false ||
                    preg_match('/<[a-z][\s\S]*>/i', substr($bodyText, 0, 500))
                );

                // If body_text is empty OR contains HTML, convert HTML to plain text
                if (empty($bodyText) || $bodyTextHasHtml) {
                    $htmlSource = !empty($bodyHtml) ? $bodyHtml : $bodyText;
                    if (!empty($htmlSource)) {
                        $bodyText = $this->htmlToPlainText($htmlSource);
                    }
                }

                // Create clean snippet
                $snippet = $messageData['snippet'];
                if (!empty($bodyText) && empty($snippet)) {
                    $snippet = substr($bodyText, 0, 250);
                } elseif (!empty($bodyHtml) && empty($snippet)) {
                    $cleanText = strip_tags($bodyHtml);
                    $cleanText = preg_replace('/\s+/', ' ', $cleanText);
                    $snippet = trim(substr($cleanText, 0, 250));
                }

                // Create or update message
                $message = Message::updateOrCreate(
                    ['provider_msg_id' => $messageData['id']],
                    [
                        'thread_id' => $thread->id,
                        'direction' => 'inbound',
                        'user_id' => $mailAccount->user_id,
                        'tenant_id' => $mailAccount->tenant_id,
                        'from_addr' => $headers['From'] ?? '',
                        'to_json' => $this->parseEmailAddresses($headers['To'] ?? ''),
                        'cc_json' => $this->parseEmailAddresses($headers['Cc'] ?? ''),
                        'bcc_json' => $this->parseEmailAddresses($headers['Bcc'] ?? ''),
                        'date' => $headers['Date'] ?? now(),
                        'snippet' => $snippet,
                        'body_text' => $bodyText,
                        'body_html' => $bodyHtml,
                    ]
                );

                if ($message->wasRecentlyCreated) {
                    $messagesCreated++;
                }

                $ingestedCount++;
            }

            Log::info('Successfully ingested OAuth emails', [
                'email' => $mailAccount->email,
                'total_processed' => $ingestedCount,
                'threads_created' => $threadsCreated,
                'messages_created' => $messagesCreated
            ]);

            return [
                'success' => true,
                'threads_created' => $threadsCreated,
                'messages_fetched' => $ingestedCount,
                'messages_created' => $messagesCreated,
                'account_email' => $mailAccount->email,
            ];

        } catch (\Exception $e) {
            Log::error('Failed to ingest OAuth emails', [
                'email' => $mailAccount->email,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Fetch emails from IMAP account
     *
     * @param MailAccount $mailAccount
     * @return array
     * @throws \Exception
     */
    private function ingestImapEmails(MailAccount $mailAccount): array
    {
        try {
            Log::info('Starting IMAP email ingestion', [
                'email' => $mailAccount->email,
                'imap_host' => $mailAccount->imap_host
            ]);

            // Fetch emails from IMAP service
            $result = $this->imapService->fetchEmails($mailAccount, 20);

            if (!$result['success']) {
                throw new \Exception($result['message']);
            }

            $emails = $result['emails'] ?? [];
            $threadsCreated = 0;
            $messagesCreated = 0;

            // Process each email
            foreach ($emails as $emailData) {
                $processed = $this->processImapEmail($emailData, $mailAccount);

                if ($processed['thread_created']) {
                    $threadsCreated++;
                }
                if ($processed['message_created']) {
                    $messagesCreated++;
                }
            }

            Log::info('Successfully ingested IMAP emails', [
                'email' => $mailAccount->email,
                'total_processed' => count($emails),
                'threads_created' => $threadsCreated,
                'messages_created' => $messagesCreated
            ]);

            return [
                'success' => true,
                'threads_created' => $threadsCreated,
                'messages_fetched' => count($emails),
                'messages_created' => $messagesCreated,
                'account_email' => $mailAccount->email,
            ];

        } catch (\Exception $e) {
            Log::error('Failed to ingest IMAP emails', [
                'email' => $mailAccount->email,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Process a single IMAP email (convert to thread/message)
     *
     * @param array $emailData
     * @param MailAccount $mailAccount
     * @return array ['thread_created' => bool, 'message_created' => bool]
     */
    private function processImapEmail(array $emailData, MailAccount $mailAccount): array
    {
        // Generate unique thread ID from IMAP message
        $threadId = 'imap_' . $mailAccount->id . '_' . ($emailData['provider_msg_id'] ?? uniqid());
        $messageId = 'imap_msg_' . $mailAccount->id . '_' . ($emailData['provider_msg_id'] ?? uniqid());

        // Create or update thread
        $thread = Thread::updateOrCreate(
            ['provider_thread_id' => $threadId],
            [
                'mail_account_id' => $mailAccount->id,
                'user_id' => $mailAccount->user_id,
                'tenant_id' => $mailAccount->tenant_id,
                'subject' => $emailData['subject'] ?? 'No Subject',
                'participants' => array_merge(
                    [$emailData['from_addr'] ?? ''],
                    $emailData['to_json'] ?? []
                ),
                'last_message_at' => $emailData['date'] ?? now(),
                'status' => 'open',
            ]
        );

        $threadCreated = $thread->wasRecentlyCreated;

        // Always update last_message_at to current time for fetched emails
        $thread->update(['last_message_at' => now()]);

        // Sanitize and process email body content
        $bodyHtml = $this->sanitizeUtf8($emailData['body_html'] ?? '');
        $bodyText = $this->sanitizeUtf8($emailData['body_text'] ?? '');

        // Check if body_text contains HTML (some IMAP servers incorrectly mark HTML as plain text)
        $bodyTextHasHtml = !empty($bodyText) && (
            stripos($bodyText, '<!DOCTYPE') !== false ||
            stripos($bodyText, '<html') !== false ||
            stripos($bodyText, '<body') !== false ||
            preg_match('/<[a-z][\s\S]*>/i', substr($bodyText, 0, 500)) // Check first 500 chars for HTML tags
        );

        // If body_text is empty OR contains HTML, convert HTML to plain text
        if (empty($bodyText) || $bodyTextHasHtml) {
            // Use body_html if available, otherwise use the HTML-containing body_text
            $htmlSource = !empty($bodyHtml) ? $bodyHtml : $bodyText;
            if (!empty($htmlSource)) {
                $bodyText = $this->htmlToPlainText($htmlSource);
            }
        }

        // Create a clean snippet (prefer body_text, fallback to stripping HTML)
        if (!empty($bodyText)) {
            $snippet = substr($bodyText, 0, 250);
        } elseif (!empty($bodyHtml)) {
            // Strip HTML tags and get first 250 chars
            $cleanText = strip_tags($bodyHtml);
            // Remove extra whitespace
            $cleanText = preg_replace('/\s+/', ' ', $cleanText);
            $snippet = trim(substr($cleanText, 0, 250));
        } else {
            $snippet = $this->sanitizeUtf8($emailData['snippet'] ?? '');
        }

        // Create or update message
        $message = Message::updateOrCreate(
            ['provider_msg_id' => $messageId],
            [
                'thread_id' => $thread->id,
                'direction' => 'inbound',
                'user_id' => $mailAccount->user_id,
                'tenant_id' => $mailAccount->tenant_id,
                'from_addr' => $emailData['from_addr'] ?? '',
                'to_json' => $emailData['to_json'] ?? [$mailAccount->email],
                'cc_json' => $emailData['cc_json'] ?? [],
                'bcc_json' => [],
                'date' => $emailData['date'] ?? now(),
                'snippet' => $snippet,
                'body_text' => $bodyText,
                'body_html' => $bodyHtml,
            ]
        );

        $messageCreated = $message->wasRecentlyCreated;

        return [
            'thread_created' => $threadCreated,
            'message_created' => $messageCreated,
        ];
    }

    /**
     * Extract participant emails from message headers
     *
     * @param array $headers
     * @return array
     */
    private function extractParticipants(array $headers): array
    {
        $participants = [];

        if (isset($headers['From'])) {
            $participants = array_merge($participants, $this->parseEmailAddresses($headers['From']));
        }

        if (isset($headers['To'])) {
            $participants = array_merge($participants, $this->parseEmailAddresses($headers['To']));
        }

        if (isset($headers['Cc'])) {
            $participants = array_merge($participants, $this->parseEmailAddresses($headers['Cc']));
        }

        return array_unique($participants);
    }

    /**
     * Parse email addresses from a header string
     *
     * @param string $addressString
     * @return array
     */
    private function parseEmailAddresses(string $addressString): array
    {
        if (empty($addressString)) {
            return [];
        }

        // Simple parsing - extract email addresses using regex
        preg_match_all('/[\w\-\.]+@[\w\-\.]+\.\w+/', $addressString, $matches);

        return $matches[0] ?? [];
    }

    /**
     * Sanitize string to remove invalid UTF-8 characters
     *
     * @param string $text
     * @return string
     */
    private function sanitizeUtf8(string $text): string
    {
        if (empty($text)) {
            return '';
        }

        // Remove invalid UTF-8 characters
        $text = mb_convert_encoding($text, 'UTF-8', 'UTF-8');

        // Remove any remaining problematic characters
        $text = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $text);

        return $text;
    }

    /**
     * Convert HTML to plain text
     *
     * @param string $html
     * @return string
     */
    private function htmlToPlainText(string $html): string
    {
        if (empty($html)) {
            return '';
        }

        // Strip all HTML tags
        $text = strip_tags($html);

        // Decode HTML entities
        $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');

        // Remove extra whitespace and newlines
        $text = preg_replace('/\s+/', ' ', $text);

        // Trim
        $text = trim($text);

        return $text;
    }
}
