<?php

namespace App\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;

class DatabaseOptimizationService
{
    /**
     * Optimize database tables
     */
    public static function optimizeTables(): array
    {
        $results = [];
        
        $tables = [
            'threads',
            'messages', 
            'email_drafts',
            'mail_accounts',
            'oauth_connections',
            'reviews',
            'email_signatures'
        ];

        foreach ($tables as $table) {
            if (Schema::hasTable($table)) {
                try {
                    DB::statement("OPTIMIZE TABLE {$table}");
                    $results[$table] = 'Optimized successfully';
                } catch (\Exception $e) {
                    $results[$table] = 'Error: ' . $e->getMessage();
                    Log::error("Failed to optimize table {$table}", ['error' => $e->getMessage()]);
                }
            }
        }

        return $results;
    }

    /**
     * Analyze table indexes
     */
    public static function analyzeIndexes(): array
    {
        $results = [];
        
        $tables = [
            'threads' => ['user_id', 'mail_account_id', 'created_at', 'last_message_at'],
            'messages' => ['thread_id', 'created_at'],
            'email_drafts' => ['thread_id', 'status', 'created_at'],
            'mail_accounts' => ['user_id', 'email', 'is_active'],
            'oauth_connections' => ['provider', 'account_email'],
            'reviews' => ['business_profile_id', 'created_at', 'rating'],
        ];

        foreach ($tables as $table => $columns) {
            if (Schema::hasTable($table)) {
                $indexes = DB::select("SHOW INDEX FROM {$table}");
                $existingIndexes = array_column($indexes, 'Column_name');
                
                $missingIndexes = array_diff($columns, $existingIndexes);
                
                $results[$table] = [
                    'existing_indexes' => $existingIndexes,
                    'missing_indexes' => $missingIndexes,
                    'recommendations' => self::getIndexRecommendations($table, $missingIndexes)
                ];
            }
        }

        return $results;
    }

    /**
     * Get index recommendations
     */
    private static function getIndexRecommendations(string $table, array $missingIndexes): array
    {
        $recommendations = [];
        
        foreach ($missingIndexes as $column) {
            $recommendations[] = "CREATE INDEX idx_{$table}_{$column} ON {$table} ({$column})";
        }

        return $recommendations;
    }

    /**
     * Clean up old data
     */
    public static function cleanupOldData(): array
    {
        $results = [];
        
        try {
            // Clean up old email drafts (older than 30 days, not approved)
            $draftsDeleted = DB::table('email_drafts')
                ->where('status', '!=', 'approved')
                ->where('created_at', '<', now()->subDays(30))
                ->delete();
            $results['old_drafts'] = "Deleted {$draftsDeleted} old drafts";

            // Clean up old OAuth tokens (older than 90 days, inactive)
            $tokensDeleted = DB::table('oauth_connections')
                ->where('updated_at', '<', now()->subDays(90))
                ->whereDoesntHave('mailAccounts')
                ->delete();
            $results['old_tokens'] = "Deleted {$tokensDeleted} old OAuth tokens";

            // Clean up old review drafts (older than 60 days)
            $reviewDraftsDeleted = DB::table('review_drafts')
                ->where('created_at', '<', now()->subDays(60))
                ->where('status', '!=', 'approved')
                ->delete();
            $results['old_review_drafts'] = "Deleted {$reviewDraftsDeleted} old review drafts";

        } catch (\Exception $e) {
            $results['error'] = $e->getMessage();
            Log::error('Database cleanup failed', ['error' => $e->getMessage()]);
        }

        return $results;
    }

    /**
     * Get database statistics
     */
    public static function getDatabaseStats(): array
    {
        $stats = [];
        
        $tables = [
            'threads',
            'messages',
            'email_drafts', 
            'mail_accounts',
            'oauth_connections',
            'reviews',
            'email_signatures'
        ];

        foreach ($tables as $table) {
            if (Schema::hasTable($table)) {
                try {
                    $count = DB::table($table)->count();
                    $size = DB::select("SELECT 
                        ROUND(((data_length + index_length) / 1024 / 1024), 2) AS 'Size in MB'
                        FROM information_schema.TABLES 
                        WHERE table_schema = DATABASE() 
                        AND table_name = '{$table}'")[0]->{'Size in MB'} ?? 0;
                    
                    $stats[$table] = [
                        'rows' => $count,
                        'size_mb' => $size
                    ];
                } catch (\Exception $e) {
                    $stats[$table] = ['error' => $e->getMessage()];
                }
            }
        }

        return $stats;
    }

    /**
     * Check for slow queries
     */
    public static function getSlowQueries(): array
    {
        try {
            $slowQueries = DB::select("
                SELECT 
                    sql_text,
                    exec_count,
                    avg_timer_wait/1000000000 as avg_time_seconds,
                    sum_timer_wait/1000000000 as total_time_seconds
                FROM performance_schema.events_statements_summary_by_digest 
                WHERE avg_timer_wait > 1000000000 
                ORDER BY avg_timer_wait DESC 
                LIMIT 10
            ");
            
            return $slowQueries;
        } catch (\Exception $e) {
            Log::error('Failed to get slow queries', ['error' => $e->getMessage()]);
            return ['error' => 'Slow query analysis not available'];
        }
    }

    /**
     * Optimize specific queries
     */
    public static function optimizeQuery(string $query): array
    {
        try {
            $explain = DB::select("EXPLAIN {$query}");
            return [
                'query' => $query,
                'explain' => $explain,
                'recommendations' => self::getQueryRecommendations($explain)
            ];
        } catch (\Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }

    /**
     * Get query optimization recommendations
     */
    private static function getQueryRecommendations(array $explain): array
    {
        $recommendations = [];
        
        foreach ($explain as $row) {
            if ($row->type === 'ALL') {
                $recommendations[] = "Consider adding an index on {$row->table}";
            }
            if ($row->Extra && strpos($row->Extra, 'Using filesort') !== false) {
                $recommendations[] = "Consider adding an ORDER BY index";
            }
            if ($row->Extra && strpos($row->Extra, 'Using temporary') !== false) {
                $recommendations[] = "Consider optimizing GROUP BY or ORDER BY";
            }
        }

        return array_unique($recommendations);
    }
}
