<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Redis;

class CacheService
{
    const CACHE_PREFIX = 'replypilot:';
    const DEFAULT_TTL = 3600; // 1 hour

    /**
     * Cache email account data
     */
    public static function cacheEmailAccount($accountId, $data, $ttl = null): void
    {
        $key = self::CACHE_PREFIX . "email_account:{$accountId}";
        Cache::put($key, $data, $ttl ?? self::DEFAULT_TTL);
    }

    /**
     * Get cached email account data
     */
    public static function getCachedEmailAccount($accountId)
    {
        $key = self::CACHE_PREFIX . "email_account:{$accountId}";
        return Cache::get($key);
    }

    /**
     * Cache thread data with pagination
     */
    public static function cacheThreads($userId, $page, $filters, $data, $ttl = null): void
    {
        $filterKey = md5(serialize($filters));
        $key = self::CACHE_PREFIX . "threads:{$userId}:{$page}:{$filterKey}";
        Cache::put($key, $data, $ttl ?? 1800); // 30 minutes
    }

    /**
     * Get cached threads
     */
    public static function getCachedThreads($userId, $page, $filters)
    {
        $filterKey = md5(serialize($filters));
        $key = self::CACHE_PREFIX . "threads:{$userId}:{$page}:{$filterKey}";
        return Cache::get($key);
    }

    /**
     * Cache AI responses
     */
    public static function cacheAIResponse($prompt, $response, $ttl = null): void
    {
        $key = self::CACHE_PREFIX . "ai_response:" . md5($prompt);
        Cache::put($key, $response, $ttl ?? 7200); // 2 hours
    }

    /**
     * Get cached AI response
     */
    public static function getCachedAIResponse($prompt)
    {
        $key = self::CACHE_PREFIX . "ai_response:" . md5($prompt);
        return Cache::get($key);
    }

    /**
     * Cache email signatures
     */
    public static function cacheSignatures($accountId, $signatures, $ttl = null): void
    {
        $key = self::CACHE_PREFIX . "signatures:{$accountId}";
        Cache::put($key, $signatures, $ttl ?? self::DEFAULT_TTL);
    }

    /**
     * Get cached signatures
     */
    public static function getCachedSignatures($accountId)
    {
        $key = self::CACHE_PREFIX . "signatures:{$accountId}";
        return Cache::get($key);
    }

    /**
     * Cache review data
     */
    public static function cacheReviews($businessId, $reviews, $ttl = null): void
    {
        $key = self::CACHE_PREFIX . "reviews:{$businessId}";
        Cache::put($key, $reviews, $ttl ?? 1800); // 30 minutes
    }

    /**
     * Get cached reviews
     */
    public static function getCachedReviews($businessId)
    {
        $key = self::CACHE_PREFIX . "reviews:{$businessId}";
        return Cache::get($key);
    }

    /**
     * Invalidate cache patterns
     */
    public static function invalidatePattern(string $pattern): void
    {
        if (config('cache.default') === 'redis') {
            $keys = Redis::keys(self::CACHE_PREFIX . $pattern);
            if (!empty($keys)) {
                Redis::del($keys);
            }
        } else {
            // For file cache, we can't easily pattern match
            // This would need to be implemented differently
            Cache::flush();
        }
    }

    /**
     * Invalidate user-specific cache
     */
    public static function invalidateUserCache($userId): void
    {
        self::invalidatePattern("threads:{$userId}:*");
        self::invalidatePattern("email_account:*");
    }

    /**
     * Cache database query results
     */
    public static function rememberQuery(string $key, callable $callback, $ttl = null)
    {
        $cacheKey = self::CACHE_PREFIX . "query:" . md5($key);
        return Cache::remember($cacheKey, $ttl ?? 1800, $callback);
    }

    /**
     * Warm up frequently accessed cache
     */
    public static function warmUpCache(): void
    {
        // Pre-load common data
        $accounts = \App\Models\MailAccount::where('is_active', true)->get();
        foreach ($accounts as $account) {
            self::cacheEmailAccount($account->id, $account->toArray());
        }

        // Pre-load signatures
        $signatureAccounts = \App\Models\MailAccount::whereHas('emailSignatures')->get();
        foreach ($signatureAccounts as $account) {
            $signatures = $account->emailSignatures()->active()->get();
            self::cacheSignatures($account->id, $signatures->toArray());
        }
    }

    /**
     * Get cache statistics
     */
    public static function getCacheStats(): array
    {
        if (config('cache.default') === 'redis') {
            $info = Redis::info();
            return [
                'used_memory' => $info['used_memory_human'] ?? 'N/A',
                'connected_clients' => $info['connected_clients'] ?? 0,
                'keyspace_hits' => $info['keyspace_hits'] ?? 0,
                'keyspace_misses' => $info['keyspace_misses'] ?? 0,
                'hit_rate' => self::calculateHitRate($info),
            ];
        }

        return ['driver' => config('cache.default')];
    }

    private static function calculateHitRate($info): float
    {
        $hits = $info['keyspace_hits'] ?? 0;
        $misses = $info['keyspace_misses'] ?? 0;
        $total = $hits + $misses;
        
        return $total > 0 ? round(($hits / $total) * 100, 2) : 0;
    }
}
