<?php

namespace App\Services;

use App\Contracts\AIProviderInterface;
use App\Services\AIProviders\OpenAIProvider;
use App\Services\AIProviders\GeminiProvider;
use App\Services\AIProviders\AnthropicProvider;
use App\Models\Tenant;
use Illuminate\Support\Facades\Log;

class AIServiceManager
{
    private array $providers = [];
    private string $defaultProvider;

    public function __construct()
    {
        $settings = $this->resolveAiSettings();
        $this->defaultProvider = $settings['default_provider'];
        $this->initializeProviders($settings['providers']);
    }

    /**
     * Initialize available AI providers
     */
    private function initializeProviders(?array $providerOverrides = null): void
    {
        $providerConfigs = config('services.ai.providers', []);

        foreach ($providerConfigs as $providerKey => $config) {
            $enabled = $providerOverrides[$providerKey]['enabled'] ?? $config['enabled'];
            if ($enabled) {
                try {
                    $this->providers[$providerKey] = $this->createProvider($providerKey);
                } catch (\Exception $e) {
                    Log::warning("Failed to initialize AI provider {$providerKey}: " . $e->getMessage());
                }
            }
        }
    }

    /**
     * Create a provider instance
     */
    private function createProvider(string $providerKey): AIProviderInterface
    {
        return match ($providerKey) {
            'openai' => new OpenAIProvider(),
            'gemini' => new GeminiProvider(),
            'anthropic' => new AnthropicProvider(),
            default => throw new \InvalidArgumentException("Unknown AI provider: {$providerKey}")
        };
    }

    /**
     * Get the default AI provider
     */
    public function getDefaultProvider(): AIProviderInterface
    {
        if (isset($this->providers[$this->defaultProvider])) {
            return $this->providers[$this->defaultProvider];
        }

        // Fallback to first available provider
        $availableProvider = array_key_first($this->providers);
        if ($availableProvider) {
            return $this->providers[$availableProvider];
        }

        throw new \RuntimeException('No AI providers are available');
    }

    /**
     * Get a specific AI provider
     */
    public function getProvider(string $providerKey): AIProviderInterface
    {
        if (!isset($this->providers[$providerKey])) {
            throw new \InvalidArgumentException("AI provider '{$providerKey}' is not available");
        }

        return $this->providers[$providerKey];
    }

    /**
     * Get all available providers
     */
    public function getAvailableProviders(): array
    {
        return $this->providers;
    }

    /**
     * Get provider configuration info
     */
    public function getProviderInfo(): array
    {
        $info = [];
        $settings = $this->resolveAiSettings();
        $providerConfigs = config('services.ai.providers', []);

        foreach ($providerConfigs as $key => $config) {
            $enabled = $settings['providers'][$key]['enabled'] ?? $config['enabled'];
            $configured = false;

            try {
                $configured = $this->createProvider($key)->isConfigured();
            } catch (\Exception $e) {
                $configured = false;
            }

            $info[$key] = [
                'name' => $config['name'],
                'description' => $config['description'],
                'enabled' => $enabled,
                'available' => true,
                'configured' => $configured,
                'is_default' => $key === $settings['default_provider'],
            ];
        }

        return $info;
    }

    /**
     * Test all available providers
     */
    public function testAllProviders(): array
    {
        $results = [];

        foreach ($this->providers as $key => $provider) {
            try {
                $results[$key] = $provider->testConnection();
            } catch (\Exception $e) {
                $results[$key] = [
                    'success' => false,
                    'error' => $e->getMessage(),
                ];
            }
        }

        return $results;
    }

    /**
     * Resolve AI settings from tenant settings or fallback to config
     */
    private function resolveAiSettings(): array
    {
        $defaultProvider = config('services.ai.default_provider', 'gemini');
        $providers = null;

        if (auth()->check()) {
            $tenant = auth()->user()->tenant;

            if (!$tenant && auth()->user()->role === 'super_admin') {
                $tenant = Tenant::where('slug', 'default')->first();
            }

            if ($tenant && is_array($tenant->settings ?? null)) {
                $aiSettings = $tenant->settings['ai_settings'] ?? null;
                if (is_array($aiSettings)) {
                    $defaultProvider = $aiSettings['default_provider'] ?? $defaultProvider;
                    $providers = $aiSettings['providers'] ?? null;
                }
            }
        }

        return [
            'default_provider' => $defaultProvider,
            'providers' => $providers,
        ];
    }
}
