<?php

namespace App\Services\AIProviders;

use App\Contracts\AIProviderInterface;
use App\Support\RetryHelper;
use App\Support\SecureHttpClient;
use OpenAI;
use Illuminate\Support\Facades\Log;

class OpenAIProvider implements AIProviderInterface
{
    private $client;

    public function __construct()
    {
        $httpClient = SecureHttpClient::create([
            'timeout' => 30,
        ]);

        $this->client = OpenAI::factory()
            ->withApiKey(config('services.openai.api_key'))
            ->withHttpClient($httpClient)
            ->make();
    }

    public function generateResponse(string $systemPrompt, string $userPrompt, array $options = []): array
    {
        try {
            // Use retry helper with exponential backoff for transient failures
            $response = RetryHelper::retry(
                fn() => $this->client->chat()->create([
                    'model' => config('services.openai.model', 'gpt-4o-mini'),
                    'messages' => [
                        [
                            'role' => 'system',
                            'content' => $systemPrompt,
                        ],
                        [
                            'role' => 'user',
                            'content' => $userPrompt,
                        ],
                    ],
                    'max_tokens' => $options['max_tokens'] ?? 200,
                    'temperature' => $options['temperature'] ?? 0.7,
                ]),
                maxAttempts: 3,
                baseDelayMs: 1000
            );

            return [
                'success' => true,
                'content' => trim($response->choices[0]->message->content),
                'provider' => 'openai',
            ];
        } catch (\Exception $e) {
            Log::error('OpenAI API error after retries: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'provider' => 'openai',
            ];
        }
    }

    public function testConnection(): array
    {
        try {
            $response = $this->client->chat()->create([
                'model' => config('services.openai.model', 'gpt-4o-mini'),
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => 'Say "OpenAI connection test successful"',
                    ],
                ],
                'max_tokens' => 20,
            ]);

            return [
                'success' => true,
                'message' => 'OpenAI connection successful',
                'response' => trim($response->choices[0]->message->content),
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getProviderName(): string
    {
        return 'ChatGPT';
    }

    public function isConfigured(): bool
    {
        return !empty(config('services.openai.api_key'));
    }
}
