<?php

namespace App\Services\AIProviders;

use App\Contracts\AIProviderInterface;
use App\Support\RetryHelper;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use App\Support\SecureHttpClient;

class GeminiProvider implements AIProviderInterface
{
    private ?string $apiKey;
    private string $model;

    public function __construct()
    {
        $this->apiKey = config('services.gemini.api_key');
        $this->model = config('services.gemini.model', 'gemini-2.0-flash-exp');
    }

    public function generateResponse(string $systemPrompt, string $userPrompt, array $options = []): array
    {
        try {
            if (!$this->apiKey) {
                throw new \Exception('Gemini API key not configured');
            }

            // Use retry helper with exponential backoff for transient failures
            $result = $this->requestContent($systemPrompt, $userPrompt, $options);
            $content = $result['content'];
            $finishReason = $result['finishReason'];
            $usageMetadata = $result['usageMetadata'];

            if (str_word_count($content) < 20) {
                $retryPrompt = $userPrompt
                    . "\n\nIMPORTANT: Reply must be 2-4 sentences, at least 40 words, "
                    . "and end with a complete sentence. Do not stop early.";
                $retryOptions = $options;
                $retryOptions['max_tokens'] = max($options['max_tokens'] ?? 1000, 200);
                $retryOptions['candidate_count'] = max($options['candidate_count'] ?? 2, 2);

                $retryResult = $this->requestContent($systemPrompt, $retryPrompt, $retryOptions);

                if (strlen($retryResult['content']) > strlen($content)) {
                    $content = $retryResult['content'];
                    $finishReason = $retryResult['finishReason'];
                    $usageMetadata = $retryResult['usageMetadata'];
                }
            }

            Log::info('Gemini API response details', [
                'finishReason' => $finishReason,
                'contentLength' => strlen($content),
                'requestedTokens' => $options['max_tokens'] ?? 800,
                'usageMetadata' => $usageMetadata,
                'promptTokenCount' => $usageMetadata['promptTokenCount'] ?? null,
                'candidatesTokenCount' => $usageMetadata['candidatesTokenCount'] ?? null,
                'totalTokenCount' => $usageMetadata['totalTokenCount'] ?? null,
            ]);

            return [
                'success' => true,
                'content' => trim($content),
                'provider' => 'gemini',
                'finish_reason' => $finishReason,
            ];
        } catch (\Exception $e) {
            Log::error('Gemini API error after retries: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'provider' => 'gemini',
            ];
        }
    }

    public function testConnection(): array
    {
        try {
            if (!$this->apiKey) {
                throw new \Exception('Gemini API key not configured');
            }

            // Use x-goog-api-key header instead of URL query parameter for security
            $response = Http::timeout(30)
                ->withOptions(SecureHttpClient::getHttpOptions())
                ->withHeaders([
                    'x-goog-api-key' => $this->apiKey,
                    'Content-Type' => 'application/json',
                ])
                ->post("https://generativelanguage.googleapis.com/v1beta/models/{$this->model}:generateContent", [
                    'contents' => [
                        [
                            'parts' => [
                                ['text' => 'Say "Gemini connection test successful"']
                            ]
                        ]
                    ],
                    'generationConfig' => [
                        'maxOutputTokens' => 20,
                    ]
                ]);

            if (!$response->successful()) {
                throw new \Exception('Gemini API request failed: ' . $response->body());
            }

            $data = $response->json();
            $parts = $data['candidates'][0]['content']['parts'] ?? [];
            $content = '';

            foreach ($parts as $part) {
                if (isset($part['text'])) {
                    $content .= $part['text'];
                }
            }

            return [
                'success' => true,
                'message' => 'Gemini connection successful',
                'response' => trim($content),
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getProviderName(): string
    {
        return 'Gemini';
    }

    public function isConfigured(): bool
    {
        return !empty($this->apiKey);
    }

    private function requestContent(string $systemPrompt, string $userPrompt, array $options): array
    {
        $response = RetryHelper::retry(
            fn() => Http::timeout(30)
                ->withOptions(SecureHttpClient::getHttpOptions())
                ->withHeaders([
                    'x-goog-api-key' => $this->apiKey,
                    'Content-Type' => 'application/json',
                ])
                ->post("https://generativelanguage.googleapis.com/v1beta/models/{$this->model}:generateContent", [
                    'systemInstruction' => [
                        'parts' => [
                            ['text' => $systemPrompt]
                        ]
                    ],
                    'contents' => [
                        [
                            'role' => 'user',
                            'parts' => [
                                ['text' => $userPrompt]
                            ]
                        ]
                    ],
                    'generationConfig' => array_filter([
                        'maxOutputTokens' => $options['max_tokens'] ?? 1000,
                        'temperature' => $options['temperature'] ?? 0.7,
                        'candidateCount' => $options['candidate_count'] ?? null,
                    ], static fn ($value) => $value !== null),
                ]),
            maxAttempts: 3,
            baseDelayMs: 1000
        );

        if (!$response->successful()) {
            throw new \Exception('Gemini API request failed: ' . $response->body());
        }

        $data = $response->json();
        $candidates = $data['candidates'] ?? [];
        $bestContent = '';
        $bestFinishReason = 'NO_CANDIDATES';

        if (empty($candidates)) {
            return [
                'content' => '',
                'finishReason' => $bestFinishReason,
                'usageMetadata' => $data['usageMetadata'] ?? [],
            ];
        }

        $bestFinishReason = $candidates[0]['finishReason'] ?? 'UNKNOWN';

        foreach ($candidates as $candidate) {
            $parts = $candidate['content']['parts'] ?? [];
            $content = '';

            foreach ($parts as $part) {
                if (isset($part['text'])) {
                    $content .= $part['text'];
                }
            }

            $content = trim($content);
            if (strlen($content) > strlen($bestContent)) {
                $bestContent = $content;
                $bestFinishReason = $candidate['finishReason'] ?? 'UNKNOWN';
            }
        }

        return [
            'content' => $bestContent,
            'finishReason' => $bestFinishReason,
            'usageMetadata' => $data['usageMetadata'] ?? [],
        ];
    }
}
