<?php

namespace App\Modules\SMTP\Controllers;

use App\Http\Controllers\Controller;
use App\Modules\SMTP\Services\SMTPService;
use App\Models\MailAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SMTPController extends Controller
{
    private SMTPService $smtpService;

    public function __construct(SMTPService $smtpService)
    {
        $this->smtpService = $smtpService;
    }

    /**
     * Display SMTP accounts
     */
    public function index()
    {
        $user = Auth::user();
        $accountsQuery = MailAccount::where('user_id', $user->id)
            ->where('connection_type', 'smtp');
        if ($user->tenant_id) {
            $accountsQuery->where('tenant_id', $user->tenant_id);
        }
        $accounts = $accountsQuery->get();

        return view('smtp::accounts.index', compact('accounts'));
    }

    /**
     * Store SMTP account
     */
    public function store(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'smtp_host' => 'required|string',
            'smtp_port' => 'required|integer',
            'smtp_username' => 'required|string',
            'smtp_password' => 'required|string',
            'smtp_encryption' => 'required|string',
            'from_name' => 'required|string',
        ]);

        $user = Auth::user();
        
        MailAccount::create([
            'user_id' => $user->id,
            'tenant_id' => $user->tenant_id,
            'email' => $request->email,
            'connection_type' => 'smtp',
            'smtp_host' => $request->smtp_host,
            'smtp_port' => $request->smtp_port,
            'smtp_username' => $request->smtp_username,
            'smtp_password' => $request->smtp_password,
            'smtp_encryption' => $request->smtp_encryption,
            'from_name' => $request->from_name,
            'is_primary' => false,
        ]);

        return redirect()->route('smtp.accounts.index')
            ->with('success', 'SMTP account added successfully!');
    }

    /**
     * Show edit form for SMTP account
     */
    public function edit(MailAccount $account)
    {
        $user = Auth::user();
        if ($account->user_id !== $user->id || ($user->tenant_id && $account->tenant_id !== $user->tenant_id)) {
            abort(403, 'Unauthorized.');
        }

        return view('smtp::accounts.edit', compact('account'));
    }

    /**
     * Update SMTP account
     */
    public function update(Request $request, MailAccount $account)
    {
        $user = Auth::user();
        if ($account->user_id !== $user->id || ($user->tenant_id && $account->tenant_id !== $user->tenant_id)) {
            abort(403, 'Unauthorized.');
        }

        $request->validate([
            'email' => 'required|email',
            'smtp_host' => 'required|string',
            'smtp_port' => 'required|integer',
            'smtp_username' => 'required|string',
            'smtp_password' => 'required|string',
            'smtp_encryption' => 'required|string',
            'from_name' => 'required|string',
        ]);

        $account->update($request->all());

        return redirect()->route('smtp.accounts.index')
            ->with('success', 'SMTP account updated successfully!');
    }

    /**
     * Delete SMTP account
     */
    public function destroy(MailAccount $account)
    {
        $user = Auth::user();
        if ($account->user_id !== $user->id || ($user->tenant_id && $account->tenant_id !== $user->tenant_id)) {
            abort(403, 'Unauthorized.');
        }

        $account->delete();
        
        return redirect()->route('smtp.accounts.index')
            ->with('success', 'SMTP account deleted successfully!');
    }

    /**
     * Test SMTP connection
     */
    public function testConnection(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:mail_accounts,id'
        ]);

        $user = Auth::user();
        $accountQuery = MailAccount::where('id', $request->account_id)
            ->where('user_id', $user->id);
        if ($user->tenant_id) {
            $accountQuery->where('tenant_id', $user->tenant_id);
        }
        $account = $accountQuery->firstOrFail();
        $result = $this->smtpService->testConnection($account);

        return response()->json($result);
    }

    /**
     * Send email via SMTP
     */
    public function sendEmail(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:mail_accounts,id',
            'to' => 'required|email',
            'subject' => 'required|string|max:255',
            'body' => 'required|string'
        ]);

        $user = Auth::user();
        $accountQuery = MailAccount::where('id', $request->account_id)
            ->where('user_id', $user->id);
        if ($user->tenant_id) {
            $accountQuery->where('tenant_id', $user->tenant_id);
        }
        $account = $accountQuery->firstOrFail();
        $success = $this->smtpService->sendEmail($account, $request->all());

        if ($success) {
            return response()->json([
                'success' => true,
                'message' => 'Email sent successfully'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Failed to send email'
        ], 500);
    }

    /**
     * Fetch emails via SMTP/IMAP
     */
    public function fetchEmails(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:mail_accounts,id'
        ]);

        $user = Auth::user();
        $accountQuery = MailAccount::where('id', $request->account_id)
            ->where('user_id', $user->id);
        if ($user->tenant_id) {
            $accountQuery->where('tenant_id', $user->tenant_id);
        }
        $account = $accountQuery->firstOrFail();
        $emails = $this->smtpService->fetchEmails($account);

        return response()->json([
            'success' => true,
            'emails' => $emails,
            'count' => count($emails)
        ]);
    }

    /**
     * Test IMAP connection
     */
    public function testImapConnection(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:mail_accounts,id'
        ]);

        $user = Auth::user();
        $accountQuery = MailAccount::where('id', $request->account_id)
            ->where('user_id', $user->id);
        if ($user->tenant_id) {
            $accountQuery->where('tenant_id', $user->tenant_id);
        }
        $account = $accountQuery->firstOrFail();
        
        try {
            $connectionString = "{{$account->imap_host}:{$account->imap_port}/{$account->imap_encryption}/novalidate-cert}INBOX";
            $inbox = imap_open($connectionString, $account->imap_username, $account->imap_password);
            
            if ($inbox) {
                imap_close($inbox);
                return response()->json([
                    'success' => true,
                    'message' => 'IMAP connection successful'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'IMAP connection failed: ' . imap_last_error()
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'IMAP connection failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Fetch emails via IMAP
     */
    public function fetchImapEmails(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:mail_accounts,id'
        ]);

        $user = Auth::user();
        $accountQuery = MailAccount::where('id', $request->account_id)
            ->where('user_id', $user->id);
        if ($user->tenant_id) {
            $accountQuery->where('tenant_id', $user->tenant_id);
        }
        $account = $accountQuery->firstOrFail();
        $emails = $this->smtpService->fetchEmails($account);

        return response()->json([
            'success' => true,
            'emails' => $emails,
            'count' => count($emails)
        ]);
    }
}
