<?php

namespace App\Modules\Reviews\Controllers;

use App\Http\Controllers\Controller;
use App\Modules\Reviews\Services\ReviewsService;
use App\Models\Review;
use App\Models\ReviewDraft;
use App\Models\Location;
use App\Services\UserContextService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class ReviewsController extends Controller
{
    private UserContextService $userContext;

    public function __construct(UserContextService $userContext)
    {
        $this->userContext = $userContext;
    }

    /**
     * Display reviews
     */
    public function index()
    {
        try {
            // Get user-scoped reviews through locations
            $userLocations = $this->userContext->getUserLocations()->get();
            $locationIds = $userLocations->pluck('id');
            
            // Query reviews directly with proper pagination
            $reviews = Review::whereIn('location_id', $locationIds)
                ->with(['location', 'drafts'])
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            return view('reviews.index', compact('reviews'));
        } catch (\Exception $e) {
            \Log::error('Reviews index error: ' . $e->getMessage());
            return view('reviews.index', ['reviews' => collect()]);
        }
    }

    /**
     * Import Google Business Profile locations
     */
    public function importLocations()
    {
        try {
            $user = Auth::user();
            $importedCount = 0;

            // Get Google Business Profile locations
            $gbpService = app(\App\Services\GoogleBusinessProfile::class);
            $gbpLocations = $gbpService->listLocations();
            
            foreach ($gbpLocations as $gbpLocation) {
                Location::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'account_id' => $gbpLocation['account_id'],
                        'location_id' => $gbpLocation['location_id'],
                    ],
                    [
                        'source' => 'google_gbp',
                        'display_name' => $gbpLocation['display_name'],
                    ]
                );
                $importedCount++;
            }

            return redirect()->route('reviews.index')
                ->with('success', "Successfully imported {$importedCount} Google Business Profile locations.");
                
        } catch (\Exception $e) {
            Log::error('Failed to import locations: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to import locations: ' . $e->getMessage());
        }
    }

    /**
     * Fetch reviews from Google Business Profile
     */
    public function ingest()
    {
        try {
            // Get user-scoped locations
            $locations = $this->userContext->getUserLocations()->get();
            
            if ($locations->count() === 0) {
                return redirect()->route('reviews.index')
                    ->with('error', 'No Google Business Profile locations found. Please import locations first.');
            }
            
            $ingestedCount = 0;
            $gbpService = app(\App\Services\GoogleBusinessProfile::class);

            foreach ($locations as $location) {
                // Extract numeric IDs from resource names
                $accountId = basename($location->account_id);
                $locationId = basename($location->location_id);
                
                $reviews = $gbpService->listReviews($accountId, $locationId);
                
                foreach ($reviews as $reviewData) {
                    Review::updateOrCreate(
                        [
                            'location_id' => $location->id,
                            'external_review_id' => $reviewData['id'],
                        ],
                        [
                            'user_id' => $location->user_id,
                            'tenant_id' => $location->tenant_id,
                            'source' => 'google_gbp',
                            'stars' => $reviewData['stars'],
                            'lang' => $reviewData['lang'] ?? 'en',
                            'text' => $reviewData['comment'],
                            'status' => 'new',
                        ]
                    );
                    $ingestedCount++;
                }
            }

            return redirect()->route('reviews.index')
                ->with('success', "Successfully ingested {$ingestedCount} reviews.");
                
        } catch (\Exception $e) {
            Log::error('Failed to ingest reviews: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to ingest reviews: ' . $e->getMessage());
        }
    }

    /**
     * Show edit form for review draft
     */
    public function edit(Review $review)
    {
        $draft = $review->latestDraft();
        
        if (!$draft) {
            return redirect()->route('reviews.index')
                ->with('error', 'No draft found for this review.');
        }
        
        return view('reviews.edit', compact('review', 'draft'));
    }

    /**
     * Update review draft
     */
    public function update(Request $request, Review $review)
    {
        $request->validate([
            'draft_reply' => 'required|string|max:2000',
        ]);

        $draft = $review->latestDraft();
        
        if (!$draft) {
            return redirect()->route('reviews.index')
                ->with('error', 'No draft found for this review.');
        }

        $draft->update([
            'draft_reply' => $request->draft_reply,
        ]);

        return redirect()->route('reviews.index')
            ->with('success', 'Draft updated successfully.');
    }

    /**
     * Generate AI draft for review
     */
    public function draft(Review $review)
    {
        try {
            $aiManager = app(\App\Services\AIServiceManager::class);
            $draftService = new \App\Services\EnhancedReplyDraftService($aiManager);
            $result = $draftService->makeForReview($review);

            if (!empty($result['meta']['error'])) {
                return redirect()->route('reviews.index')
                    ->with('error', 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error'));
            }
            
            $draft = ReviewDraft::create([
                'review_id' => $review->id,
                'user_id' => $review->user_id,
                'tenant_id' => $review->tenant_id,
                'draft_reply' => $result['draft'],
                'ai_meta' => $result['meta'],
                'status' => 'pending',
            ]);

            $review->update(['status' => 'drafted']);

            return redirect()->route('reviews.index')
                ->with('success', 'AI draft generated successfully');
                
        } catch (\Exception $e) {
            Log::error('Failed to generate review draft: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to generate draft: ' . $e->getMessage());
        }
    }

    /**
     * Approve and post review reply
     */
    public function approve(Review $review)
    {
        try {
            $draft = $review->latestDraft();
            
            if (!$draft) {
                return redirect()->route('reviews.index')
                    ->with('error', 'No draft found for this review');
            }

            $gbpService = app(\App\Services\GoogleBusinessProfile::class);
            $result = $gbpService->replyToReview(
                $review->location->account_id,
                $review->location->location_id,
                $review->external_review_id,
                $draft->draft_reply
            );
            
            if ($result['success']) {
                $draft->update(['status' => 'sent']);
                $review->update([
                    'status' => 'posted',
                    'reviewed_at' => now(),
                ]);
                
                return redirect()->route('reviews.index')
                    ->with('success', 'Reply posted successfully');
            } else {
                return redirect()->route('reviews.index')
                    ->with('error', 'Failed to post reply: ' . $result['message']);
            }
        } catch (\Exception $e) {
            Log::error('Failed to post review reply: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to post reply: ' . $e->getMessage());
        }
    }

    /**
     * Post review reply (alias for approve)
     */
    public function post(Review $review)
    {
        return $this->approve($review);
    }

    /**
     * Delete review
     */
    public function deleteReview(Review $review)
    {
        try {
            $review->delete();
            
            return redirect()->route('reviews.index')
                ->with('success', 'Review deleted successfully.');
                
        } catch (\Exception $e) {
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to delete review: ' . $e->getMessage());
        }
    }

    /**
     * Delete review draft
     */
    public function deleteDraft(ReviewDraft $reviewDraft)
    {
        try {
            $review = $reviewDraft->review;
            $reviewDraft->delete();

            if ($review && $review->status === 'drafted' && ! $review->drafts()->exists()) {
                $review->update(['status' => 'new']);
            }
            
            return redirect()->route('reviews.index')
                ->with('success', 'Draft deleted successfully.');
                
        } catch (\Exception $e) {
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to delete draft: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete reviews
     */
    public function bulkDelete(Request $request)
    {
        if (!Auth::user()->isAdmin()) {
            abort(403, 'Unauthorized. Admin access required.');
        }
        
        $reviewIds = $request->input('review_ids', []);
        
        if (empty($reviewIds)) {
            return redirect()->route('reviews.index')
                ->with('error', 'No reviews selected for deletion.');
        }
        
        // Get user-scoped reviews to ensure we only delete reviews the user has access to
        $userLocations = $this->userContext->getUserLocations()->get();
        $locationIds = $userLocations->pluck('id');
        
        $deletedCount = Review::whereIn('id', $reviewIds)
            ->whereIn('location_id', $locationIds)
            ->count();
            
        Review::whereIn('id', $reviewIds)
            ->whereIn('location_id', $locationIds)
            ->delete();
        
        return redirect()->route('reviews.index')
            ->with('success', "Successfully deleted {$deletedCount} review(s) and all associated data.");
    }
}
