<?php

namespace App\Modules\Gmail\Services;

use App\Models\MailAccount;
use App\Models\OauthConnection;
use App\Support\GoogleClientFactory;
use App\Support\Base64Url;
use Illuminate\Support\Facades\Log;

/**
 * Gmail module service for OAuth-based email operations.
 *
 * Note: For new code, prefer using App\Services\GmailService which has
 * more complete functionality including retry logic and better error handling.
 * This module service is maintained for backward compatibility with the
 * Gmail module's controller.
 *
 * @see \App\Services\GmailService
 */
class GmailService
{
    private GoogleClientFactory $clientFactory;

    public function __construct(GoogleClientFactory $clientFactory)
    {
        $this->clientFactory = $clientFactory;
    }

    /**
     * Get Gmail service instance
     */
    public function getGmailService()
    {
        return $this->clientFactory->getGmailService();
    }

    /**
     * Fetch emails from Gmail
     */
    public function fetchEmails(MailAccount $account, int $limit = 50): array
    {
        try {
            $gmail = $this->getGmailService();
            $messages = $gmail->users_messages->listUsersMessages('me', [
                'maxResults' => $limit,
                'q' => 'in:inbox'
            ]);

            $emails = [];
            foreach ($messages->getMessages() as $message) {
                $msg = $gmail->users_messages->get('me', $message->getId());
                $emails[] = $this->parseGmailMessage($msg);
            }

            return $emails;
        } catch (\Exception $e) {
            Log::error('Gmail fetch error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Parse Gmail message into standardized format
     */
    private function parseGmailMessage($message): array
    {
        $headers = $message->getPayload()->getHeaders();
        $subject = '';
        $from = '';
        $date = '';

        foreach ($headers as $header) {
            switch ($header->getName()) {
                case 'Subject':
                    $subject = $header->getValue();
                    break;
                case 'From':
                    $from = $header->getValue();
                    break;
                case 'Date':
                    $date = $header->getValue();
                    break;
            }
        }

        return [
            'id' => $message->getId(),
            'subject' => $subject,
            'from' => $from,
            'date' => $date,
            'snippet' => $message->getSnippet(),
            'thread_id' => $message->getThreadId(),
        ];
    }

    /**
     * Send email via Gmail
     */
    public function sendEmail(MailAccount $account, array $emailData): bool
    {
        try {
            $gmail = $this->getGmailService();
            
            $message = $this->createGmailMessage($emailData);
            $gmail->users_messages->send('me', $message);
            
            return true;
        } catch (\Exception $e) {
            Log::error('Gmail send error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Create Gmail message format
     *
     * Uses URL-safe base64 encoding as required by Gmail API (RFC 4648)
     */
    private function createGmailMessage(array $emailData): \Google\Service\Gmail\Message
    {
        $message = new \Google\Service\Gmail\Message();

        $rawMessage = "To: {$emailData['to']}\r\n";
        $rawMessage .= "Subject: {$emailData['subject']}\r\n";
        $rawMessage .= "MIME-Version: 1.0\r\n";
        $rawMessage .= "Content-Type: text/plain; charset=UTF-8\r\n";
        $rawMessage .= "\r\n{$emailData['body']}";

        // Gmail API requires base64url encoding, not standard base64
        $message->setRaw(Base64Url::encode($rawMessage));

        return $message;
    }
}
