<?php

namespace App\Modules\Gmail\Controllers;

use App\Http\Controllers\Controller;
use App\Modules\Gmail\Services\GmailService;
use App\Models\MailAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Support\SecureHttpClient;

class GmailController extends Controller
{
    private GmailService $gmailService;

    public function __construct(GmailService $gmailService)
    {
        $this->gmailService = $gmailService;
    }

    /**
     * Start Gmail OAuth flow
     */
    public function startOAuth()
    {
        try {
            $config = config('services.google');
            
            // Check if Google OAuth is configured
            if (empty($config['client_id']) || empty($config['client_secret'])) {
                return redirect()->route('email.accounts')->with('error', 
                    'Google OAuth is not configured. Please add GOOGLE_CLIENT_ID and GOOGLE_CLIENT_SECRET to your .env file.'
                );
            }
            
            $client = new \Google\Client();
            $client->setClientId($config['client_id']);
            $client->setClientSecret($config['client_secret']);
            $client->setRedirectUri($config['redirect']);
            
            // Set scopes
            $scopes = explode(',', $config['scopes']);
            $client->setScopes($scopes);
            
            // Set offline access to get refresh token
            $client->setAccessType('offline');
            $client->setPrompt('consent');
            
            // Configure HTTP client with environment-aware SSL settings
            $client->setHttpClient(SecureHttpClient::create());
            
            $authUrl = $client->createAuthUrl();

            return redirect($authUrl);
        } catch (\Exception $e) {
            return redirect()->route('email.accounts')
                ->with('error', 'Failed to start Gmail OAuth: ' . $e->getMessage());
        }
    }

    /**
     * Handle Gmail OAuth callback
     */
    public function handleCallback(Request $request)
    {
        try {
            $code = $request->get('code');
            
            if (!$code) {
                return redirect()->route('email.accounts')->with('error', 'Authorization failed. No code received.');
            }

            $config = config('services.google');
            
            $client = new \Google\Client();
            $client->setClientId($config['client_id']);
            $client->setClientSecret($config['client_secret']);
            $client->setRedirectUri($config['redirect']);
            
            // Configure HTTP client with environment-aware SSL settings
            $client->setHttpClient(SecureHttpClient::create());
            
            $token = $client->fetchAccessTokenWithAuthCode($code);

            if (isset($token['error'])) {
                return redirect()->route('email.accounts')
                    ->with('error', 'Failed to exchange authorization code: ' . $token['error']);
            }

            $client->setAccessToken($token);

            // Get user profile to get email address
            $gmail = new \Google\Service\Gmail($client);
            $profile = $gmail->users->getProfile('me');
            $email = $profile->getEmailAddress();

            // Store OAuth connection
            $user = Auth::user();
            $oauthConnection = \App\Models\OauthConnection::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'tenant_id' => $user->tenant_id,
                    'provider' => 'google',
                    'account_email' => $email,
                ],
                [
                    'tokens' => $token,
                    'scopes' => explode(',', $config['scopes']),
                ]
            );

            // Create or update mail account
            $mailAccount = \App\Models\MailAccount::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'tenant_id' => $user->tenant_id,
                    'email' => $email,
                ],
                [
                    'oauth_connection_id' => $oauthConnection->id,
                    'connection_type' => 'oauth',
                    'from_name' => $user->name,
                    'is_primary' => false,
                ]
            );

            return redirect()->route('email.accounts')
                ->with('success', 'Gmail account connected successfully!');
                
        } catch (\Exception $e) {
            \Log::error('Gmail OAuth callback failed: ' . $e->getMessage());
            return redirect()->route('email.accounts')
                ->with('error', 'Failed to connect Gmail account: ' . $e->getMessage());
        }
    }

    /**
     * Display Gmail emails
     */
    public function index()
    {
        $user = Auth::user();
        $accounts = MailAccount::where('user_id', $user->id)
                              ->where('connection_type', 'oauth')
                              ->get();

        return view('gmail::emails.index', compact('accounts'));
    }

    /**
     * Fetch emails from Gmail
     */
    public function fetchEmails(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:mail_accounts,id'
        ]);

        $user = Auth::user();
        $accountQuery = MailAccount::where('id', $request->account_id)
            ->where('user_id', $user->id)
            ->where('connection_type', 'oauth');
        if ($user->tenant_id) {
            $accountQuery->where('tenant_id', $user->tenant_id);
        }
        $account = $accountQuery->firstOrFail();
        $emails = $this->gmailService->fetchEmails($account);

        return response()->json([
            'success' => true,
            'emails' => $emails,
            'count' => count($emails)
        ]);
    }

    /**
     * Send email via Gmail
     */
    public function sendEmail(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:mail_accounts,id',
            'to' => 'required|email',
            'subject' => 'required|string|max:255',
            'body' => 'required|string'
        ]);

        $user = Auth::user();
        $accountQuery = MailAccount::where('id', $request->account_id)
            ->where('user_id', $user->id)
            ->where('connection_type', 'oauth');
        if ($user->tenant_id) {
            $accountQuery->where('tenant_id', $user->tenant_id);
        }
        $account = $accountQuery->firstOrFail();
        $success = $this->gmailService->sendEmail($account, $request->all());

        if ($success) {
            return response()->json([
                'success' => true,
                'message' => 'Email sent successfully'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Failed to send email'
        ], 500);
    }

    /**
     * Display Gmail accounts
     */
    public function accounts()
    {
        $user = Auth::user();
        $accounts = MailAccount::where('user_id', $user->id)
                              ->where('connection_type', 'oauth')
                              ->get();

        return view('gmail::accounts.index', compact('accounts'));
    }

    /**
     * Store Gmail account
     */
    public function storeAccount(Request $request)
    {
        // Implementation for storing Gmail account
        return redirect()->route('gmail.accounts.index')
            ->with('success', 'Gmail account added successfully!');
    }

    /**
     * Delete Gmail account
     */
    public function deleteAccount(MailAccount $account)
    {
        $account->delete();
        
        return redirect()->route('gmail.accounts.index')
            ->with('success', 'Gmail account deleted successfully!');
    }
}
