<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\BelongsToUser;

class Thread extends Model
{
    use HasUuids, BelongsToUser, SoftDeletes;

    protected $fillable = [
        'user_id',
        'tenant_id',
        'mail_account_id',
        'provider_thread_id',
        'subject',
        'participants',
        'last_message_at',
        'status',
    ];

    protected function casts(): array
    {
        return [
            'participants' => 'array',
            'last_message_at' => 'datetime',
        ];
    }

    /**
     * User that owns this thread
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Tenant that owns this thread
     */
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Mail account this thread belongs to
     */
    public function mailAccount(): BelongsTo
    {
        return $this->belongsTo(MailAccount::class);
    }

    /**
     * Messages in this thread
     */
    public function messages(): HasMany
    {
        return $this->hasMany(Message::class);
    }

    /**
     * Email drafts for this thread
     */
    public function emailDrafts(): HasMany
    {
        return $this->hasMany(EmailDraft::class);
    }

    /**
     * Get the latest inbound message
     */
    public function latestInboundMessage()
    {
        return $this->messages()->where('direction', 'inbound')->latest()->first();
    }
}
