<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\BelongsToUser;

class MailAccount extends Model
{
    use HasUuids, BelongsToUser, SoftDeletes;

    protected $fillable = [
        'user_id',
        'tenant_id',
        'oauth_connection_id',
        'email',
        'connection_type',
        'from_name',
        'signature_html',
        'is_primary',
        'is_active',
        // SMTP fields
        'smtp_host',
        'smtp_port',
        'smtp_encryption',
        'smtp_username',
        'smtp_password',
        'smtp_from_name',
        // IMAP fields (for future)
        'imap_host',
        'imap_port',
        'imap_encryption',
        'imap_username',
        'imap_password',
        'imap_folder',
    ];

    protected function casts(): array
    {
        return [
            'is_primary' => 'boolean',
            'is_active' => 'boolean',
            'smtp_port' => 'integer',
            'smtp_password' => 'encrypted',
            'imap_port' => 'integer',
            'imap_password' => 'encrypted',
        ];
    }

    /**
     * User that owns this mail account
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Tenant that owns this mail account
     */
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * OAuth connection for this mail account
     */
    public function oauthConnection(): BelongsTo
    {
        return $this->belongsTo(OauthConnection::class);
    }

    /**
     * Threads for this mail account
     */
    public function threads(): HasMany
    {
        return $this->hasMany(Thread::class);
    }

    /**
     * Email signatures for this mail account
     */
    public function emailSignatures(): HasMany
    {
        return $this->hasMany(EmailSignature::class);
    }

    /**
     * Check if this is an OAuth account
     */
    public function isOAuth(): bool
    {
        return $this->connection_type === 'oauth' || empty($this->connection_type);
    }

    /**
     * Check if this is an SMTP account
     */
    public function isSmtp(): bool
    {
        return $this->connection_type === 'smtp';
    }

    /**
     * Get connection type display name
     */
    public function getConnectionTypeDisplayAttribute(): string
    {
        return match($this->connection_type) {
            'oauth' => 'Gmail (OAuth)',
            'smtp' => 'SMTP',
            default => 'Gmail (OAuth)'
        };
    }

    /**
     * Get SMTP configuration array
     */
    public function getSmtpConfigAttribute(): array
    {
        if (!$this->isSmtp()) {
            return [];
        }

        return [
            'host' => $this->smtp_host,
            'port' => $this->smtp_port,
            'encryption' => $this->smtp_encryption,
            'username' => $this->smtp_username,
            'password' => $this->smtp_password,
        ];
    }

    /**
     * Get IMAP configuration array
     */
    public function getImapConfigAttribute(): array
    {
        return [
            'host' => $this->imap_host,
            'port' => $this->imap_port,
            'encryption' => $this->imap_encryption,
            'username' => $this->imap_username,
            'password' => $this->imap_password,
            'folder' => $this->imap_folder ?? 'INBOX',
        ];
    }

    /**
     * Check if IMAP is configured for this account
     */
    public function hasImapConfig(): bool
    {
        return !empty($this->imap_host) && 
               !empty($this->imap_port) && 
               !empty($this->imap_username) && 
               !empty($this->imap_password);
    }
}
