<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\BelongsToUser;

class EmailDraft extends Model
{
    use HasUuids, BelongsToUser, SoftDeletes;

    protected $fillable = [
        'user_id',
        'tenant_id',
        'thread_id',
        'in_reply_to_message_id',
        'subject',
        'draft_html',
        'ai_meta',
        'status',
        'approved_by_user_id',
        'approved_at',
    ];

    protected function casts(): array
    {
        return [
            'ai_meta' => 'array',
            'approved_at' => 'datetime',
        ];
    }

    /**
     * User that owns this draft
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Tenant that owns this draft
     */
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Thread this draft belongs to
     */
    public function thread(): BelongsTo
    {
        return $this->belongsTo(Thread::class);
    }

    /**
     * Message this draft replies to
     */
    public function inReplyToMessage(): BelongsTo
    {
        return $this->belongsTo(Message::class, 'in_reply_to_message_id');
    }

    /**
     * User who approved this draft
     */
    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by_user_id');
    }

    /**
     * Increment AI regeneration count
     */
    public function incrementAiRegeneration(): void
    {
        $this->increment('ai_regeneration_count');
    }

    /**
     * Create a new version of this draft
     */
    public function createNewVersion(): self
    {
        $newDraft = $this->replicate();
        $newDraft->version_number = $this->version_number + 1;
        $newDraft->status = 'pending';
        $newDraft->approved_by_user_id = null;
        $newDraft->approved_at = null;
        $newDraft->save();
        
        return $newDraft;
    }
}
