<?php

namespace App\Jobs;

use App\Models\MailAccount;
use App\Services\IMAPEmailService;
use App\Services\GmailService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessEmailIngestion implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $timeout = 300; // 5 minutes
    public $tries = 3;
    public $backoff = [60, 120, 300]; // Retry after 1min, 2min, 5min

    protected $mailAccount;
    protected $ingestionType;

    /**
     * Create a new job instance.
     */
    public function __construct(MailAccount $mailAccount, string $ingestionType = 'imap')
    {
        $this->mailAccount = $mailAccount;
        $this->ingestionType = $ingestionType;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        Log::info('Starting email ingestion job', [
            'account_id' => $this->mailAccount->id,
            'account_email' => $this->mailAccount->email,
            'ingestion_type' => $this->ingestionType
        ]);

        try {
            if ($this->ingestionType === 'imap' && $this->mailAccount->hasImapConfig()) {
                $this->processImapIngestion();
            } elseif ($this->ingestionType === 'oauth') {
                $this->processOAuthIngestion();
            } else {
                throw new \Exception('Invalid ingestion type or missing configuration');
            }

            Log::info('Email ingestion job completed successfully', [
                'account_id' => $this->mailAccount->id,
                'account_email' => $this->mailAccount->email
            ]);

        } catch (\Exception $e) {
            Log::error('Email ingestion job failed', [
                'account_id' => $this->mailAccount->id,
                'account_email' => $this->mailAccount->email,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts()
            ]);

            throw $e; // Re-throw to trigger retry mechanism
        }
    }

    /**
     * Process IMAP email ingestion
     */
    private function processImapIngestion(): void
    {
        $imapService = app(IMAPEmailService::class);
        $imapService->fetchEmails($this->mailAccount);
    }

    /**
     * Process OAuth email ingestion
     */
    private function processOAuthIngestion(): void
    {
        $gmailService = app(GmailService::class);
        $gmailService->fetchEmails($this->mailAccount);
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('Email ingestion job permanently failed', [
            'account_id' => $this->mailAccount->id,
            'account_email' => $this->mailAccount->email,
            'error' => $exception->getMessage(),
            'attempts' => $this->attempts()
        ]);

        // Optionally notify administrators or update account status
        $this->mailAccount->update([
            'last_error' => $exception->getMessage(),
            'last_error_at' => now()
        ]);
    }
}
