<?php

namespace App\Http\Requests;

use App\Services\SecurityService;
use Illuminate\Foundation\Http\FormRequest;

class SecureSmtpRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'email' => [
                'required',
                'email',
                'max:255',
            ],
            'from_name' => [
                'required',
                'string',
                'max:100',
                'regex:/^[a-zA-Z0-9\s\-_.]*$/',
            ],
            'smtp_host' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-zA-Z0-9\-_.]+$/',
            ],
            'smtp_port' => [
                'required',
                'integer',
                'min:1',
                'max:65535',
            ],
            'smtp_username' => [
                'required',
                'string',
                'max:255',
                'email',
            ],
            'smtp_password' => [
                'required',
                'string',
                'min:6',
                'max:255',
            ],
            'smtp_encryption' => [
                'required',
                'string',
                'in:tls,ssl,none',
            ],
            'imap_host' => [
                'sometimes',
                'string',
                'max:255',
                'regex:/^[a-zA-Z0-9\-_.]+$/',
            ],
            'imap_port' => [
                'sometimes',
                'integer',
                'min:1',
                'max:65535',
            ],
            'imap_username' => [
                'sometimes',
                'string',
                'max:255',
                'email',
            ],
            'imap_password' => [
                'sometimes',
                'string',
                'min:6',
                'max:255',
            ],
        ];
    }

    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'from_name.regex' => 'From name contains invalid characters.',
            'smtp_host.regex' => 'SMTP host contains invalid characters.',
            'imap_host.regex' => 'IMAP host contains invalid characters.',
            'smtp_encryption.in' => 'Invalid encryption type.',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator($validator): void
    {
        $validator->after(function ($validator) {
            // Check for SQL injection in all string fields
            foreach ($this->all() as $key => $value) {
                if (is_string($value) && SecurityService::detectSqlInjection($value)) {
                    $validator->errors()->add($key, 'Invalid input detected.');
                }
            }

            // Check for XSS in string fields
            foreach ($this->all() as $key => $value) {
                if (is_string($value) && SecurityService::detectXss($value)) {
                    $validator->errors()->add($key, 'Invalid content detected.');
                }
            }
        });
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        // Sanitize input data
        $this->merge([
            'from_name' => SecurityService::sanitizeInput($this->from_name ?? ''),
            'smtp_host' => SecurityService::sanitizeInput($this->smtp_host ?? ''),
            'imap_host' => SecurityService::sanitizeInput($this->imap_host ?? ''),
        ]);
    }
}
