<?php

namespace App\Http\Requests;

use App\Services\SecurityService;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class SecureEmailRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return [
            'subject' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-zA-Z0-9\s\-_.,!?@#$%^&*()+=\[\]{}|\\:";\'<>\/]*$/',
            ],
            'draft_html' => [
                'required',
                'string',
                'max:10000',
            ],
            'account_id' => [
                'required',
                'string',
                'uuid',
            ],
            'thread_id' => [
                'sometimes',
                'string',
                'uuid',
            ],
        ];
    }

    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'subject.regex' => 'Subject contains invalid characters.',
            'draft_html.max' => 'Email content is too long.',
            'account_id.uuid' => 'Invalid account ID format.',
            'thread_id.uuid' => 'Invalid thread ID format.',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator($validator): void
    {
        $validator->after(function ($validator) {
            // Check for SQL injection in all string fields
            foreach ($this->all() as $key => $value) {
                if (is_string($value) && SecurityService::detectSqlInjection($value)) {
                    $validator->errors()->add($key, 'Invalid input detected.');
                }
            }

            // Check for XSS in HTML content
            if ($this->has('draft_html') && SecurityService::detectXss($this->draft_html)) {
                $validator->errors()->add('draft_html', 'Invalid HTML content detected.');
            }
        });
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        // Sanitize input data
        $this->merge([
            'subject' => SecurityService::sanitizeInput($this->subject ?? ''),
            'draft_html' => SecurityService::sanitizeEmailContent($this->draft_html ?? ''),
        ]);
    }
}
