<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Symfony\Component\HttpFoundation\Response;

class RateLimitApiCalls
{
    /**
     * Rate limit configuration for different API types
     */
    private array $limits = [
        'ai' => [
            'max_attempts' => 60,      // 60 requests
            'decay_minutes' => 1,      // per minute
        ],
        'google' => [
            'max_attempts' => 100,     // 100 requests
            'decay_minutes' => 1,      // per minute
        ],
        'email' => [
            'max_attempts' => 30,      // 30 requests
            'decay_minutes' => 1,      // per minute
        ],
    ];

    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, string $type = 'ai'): Response
    {
        $user = $request->user();

        if (!$user) {
            return $next($request);
        }

        $key = $this->resolveRequestKey($user->id, $type);
        $limits = $this->limits[$type] ?? $this->limits['ai'];

        if (RateLimiter::tooManyAttempts($key, $limits['max_attempts'])) {
            $seconds = RateLimiter::availableIn($key);

            return response()->json([
                'error' => 'Too many requests',
                'message' => "Rate limit exceeded. Please try again in {$seconds} seconds.",
                'retry_after' => $seconds,
            ], 429);
        }

        RateLimiter::hit($key, $limits['decay_minutes'] * 60);

        $response = $next($request);

        // Add rate limit headers
        $response->headers->set('X-RateLimit-Limit', $limits['max_attempts']);
        $response->headers->set('X-RateLimit-Remaining', RateLimiter::remaining($key, $limits['max_attempts']));

        return $response;
    }

    /**
     * Resolve the request key for rate limiting
     */
    private function resolveRequestKey(string $userId, string $type): string
    {
        return "rate_limit:{$type}:{$userId}";
    }
}
