<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Tenant;
use App\Services\UserContextService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Validation\Rule;

class UserManagementController extends Controller
{
    use AuthorizesRequests;
    
    private UserContextService $userContext;

    public function __construct(UserContextService $userContext)
    {
        $this->userContext = $userContext;
    }

    /**
     * Display user management page
     */
    public function index()
    {
        $user = auth()->user();
        
        if ($user->isSuperAdmin()) {
            // Super admin can see all users
            $users = User::with('tenant')->get();
            $tenants = Tenant::all();
        } else {
            // Tenant admin can only see users in their tenant
            $users = User::where('tenant_id', $user->tenant_id)->with('tenant')->get();
            $tenants = collect([$user->tenant]);
        }

        return view('user-management.index', compact('users', 'tenants'));
    }

    /**
     * Show create user form
     */
    public function create()
    {
        $user = auth()->user();
        
        if ($user->isSuperAdmin()) {
            $tenants = Tenant::all();
        } else {
            $tenants = collect([$user->tenant]);
        }

        return view('user-management.create', compact('tenants'));
    }

    /**
     * Store new user
     */
    public function store(Request $request)
    {
        $user = auth()->user();
        
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:agent,tenant_admin',
            'tenant_id' => 'required|exists:tenants,id',
        ]);

        // Check permissions
        if (!$user->isSuperAdmin() && $request->tenant_id != $user->tenant_id) {
            return redirect()->back()->with('error', 'You can only create users for your own tenant.');
        }

        if ($user->isAgent() && $request->role === 'tenant_admin') {
            return redirect()->back()->with('error', 'You cannot create tenant admin users.');
        }

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'tenant_id' => $request->tenant_id,
            'is_active' => true,
            'settings' => [
                'theme' => 'light',
                'notifications' => true,
            ],
        ]);

        return redirect()->route('user-management.index')
            ->with('success', 'User created successfully.');
    }

    /**
     * Show edit user form
     */
    public function edit(User $user)
    {
        $currentUser = auth()->user();
        
        // Check permissions
        if (!$currentUser->isSuperAdmin() && $user->tenant_id != $currentUser->tenant_id) {
            return redirect()->route('user-management.index')
                ->with('error', 'You can only edit users in your own tenant.');
        }

        if ($currentUser->isSuperAdmin()) {
            $tenants = Tenant::all();
        } else {
            $tenants = collect([$currentUser->tenant]);
        }

        return view('user-management.edit', compact('user', 'tenants'));
    }

    /**
     * Update user
     */
    public function update(Request $request, User $user)
    {
        $currentUser = auth()->user();
        
        // Check permissions
        if (!$currentUser->isSuperAdmin() && $user->tenant_id != $currentUser->tenant_id) {
            return redirect()->route('user-management.index')
                ->with('error', 'You can only edit users in your own tenant.');
        }

        $allowedRoles = ['agent', 'tenant_admin'];
        if ($currentUser->isSuperAdmin()) {
            $allowedRoles[] = 'super_admin';
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'role' => ['required', Rule::in($allowedRoles)],
            'tenant_id' => [
                Rule::requiredIf($request->input('role') !== 'super_admin'),
                'nullable',
                'exists:tenants,id',
            ],
            'password' => 'nullable|string|min:8|confirmed',
            'is_active' => 'boolean',
        ]);

        if (!$currentUser->isSuperAdmin() && $request->tenant_id != $currentUser->tenant_id) {
            return redirect()->back()->with('error', 'You can only assign users to your own tenant.');
        }

        if ($currentUser->isAgent() && $request->role === 'tenant_admin') {
            return redirect()->back()->with('error', 'You cannot change users to tenant admin role.');
        }

        $tenantId = $request->input('role') === 'super_admin' ? null : $request->tenant_id;

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'role' => $request->role,
            'tenant_id' => $tenantId,
            'is_active' => $request->boolean('is_active'),
        ]);

        if ($request->filled('password')) {
            $user->update([
                'password' => Hash::make($request->password),
            ]);
        }

        return redirect()->route('user-management.index')
            ->with('success', 'User updated successfully.');
    }

    /**
     * Delete user
     */
    public function destroy(User $user)
    {
        $currentUser = auth()->user();
        
        // Check permissions
        if (!$currentUser->isSuperAdmin() && $user->tenant_id != $currentUser->tenant_id) {
            return redirect()->route('user-management.index')
                ->with('error', 'You can only delete users in your own tenant.');
        }

        if ($user->id === $currentUser->id) {
            return redirect()->route('user-management.index')
                ->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        return redirect()->route('user-management.index')
            ->with('success', 'User deleted successfully.');
    }
}
