<?php

namespace App\Http\Controllers\TenantAdmin;

use App\Http\Controllers\Controller;
use App\Models\Location;
use App\Models\Review;
use App\Models\ReviewDraft;
use App\Models\OauthConnection;
use App\Services\GoogleBusinessProfile;
use App\Services\EnhancedReplyDraftService;
use App\Services\AIServiceManager;
use App\Services\UserContextService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReviewController extends Controller
{
    private UserContextService $userContext;

    public function __construct(UserContextService $userContext)
    {
        $this->userContext = $userContext;
    }

    /**
     * Display reviews - Tenant Admin sees reviews within their tenant
     */
    public function index()
    {
        try {
            // Tenant Admin sees reviews within their tenant
            $userLocations = $this->userContext->getUserLocations()->get();
            $locationIds = $userLocations->pluck('id');
            
            $reviews = Review::whereIn('location_id', $locationIds)
                ->with(['location', 'drafts'])
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            $googleConnections = OauthConnection::where('user_id', auth()->id())
                ->where('tenant_id', auth()->user()->tenant_id)
                ->where('provider', 'google')
                ->get();
            $aiManager = app(AIServiceManager::class);
            $providerInfo = $aiManager->getProviderInfo();
            $defaultKey = collect($providerInfo)->filter(fn ($info) => $info['is_default'])->keys()->first();
            $defaultProvider = $defaultKey ? $providerInfo[$defaultKey] : null;

            $status = [
                'oauth_connected' => $googleConnections->isNotEmpty(),
                'oauth_emails' => $googleConnections->pluck('account_email')->filter()->values()->all(),
                'locations_count' => $userLocations->count(),
                'reviews_count' => $reviews->total(),
                'ai_default_name' => $defaultProvider['name'] ?? 'Not set',
                'ai_default_ready' => $defaultProvider['configured'] ?? false,
            ];

            return view('tenant-admin.reviews.index', compact('reviews', 'status'));
        } catch (\Exception $e) {
            Log::error('Tenant Admin reviews index error: ' . $e->getMessage());
            return view('tenant-admin.reviews.index', ['reviews' => collect()]);
        }
    }

    public function disconnectGoogle()
    {
        $user = Auth::user();

        if (!$user) {
            return redirect()->route('tenant-admin.reviews.index')
                ->with('error', 'You must be logged in to disconnect Google Business Profile.');
        }

        DB::transaction(function () use ($user) {
            $reviewsQuery = Review::where('user_id', $user->id)
                ->where('tenant_id', $user->tenant_id);

            $reviewIds = $reviewsQuery->pluck('id');
            ReviewDraft::whereIn('review_id', $reviewIds)->delete();
            $reviewsQuery->delete();

            Location::where('user_id', $user->id)
                ->where('tenant_id', $user->tenant_id)
                ->delete();

            OauthConnection::where('provider', 'google')
                ->where('user_id', $user->id)
                ->where('tenant_id', $user->tenant_id)
                ->delete();
        });

        return redirect()->route('tenant-admin.reviews.index')
            ->with('success', 'Disconnected Google Business Profile and removed associated data.');
    }

    /**
     * Import locations - Tenant Admin can import locations for their tenant
     */
    public function importLocations()
    {
        try {
            $user = Auth::user();
            $importedCount = 0;
            
            $gbpService = app(GoogleBusinessProfile::class);
            $gbpLocations = $gbpService->listLocations();
            
            if (empty($gbpLocations)) {
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('error', 'No Google Business Profile locations found for your account. Please connect your Google Business Profile first.');
            }

            foreach ($gbpLocations as $gbpLocation) {
                Location::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'account_id' => $gbpLocation['account_id'],
                        'location_id' => $gbpLocation['location_id'],
                    ],
                    [
                        'source' => 'google_gbp',
                        'display_name' => $gbpLocation['display_name'],
                    ]
                );
                $importedCount++;
            }

            if ($importedCount === 0) {
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('error', 'No Google Business Profile locations found for your account. Please connect your Google Business Profile first.');
            }

            Log::info('Tenant reviews: imported locations', [
                'user_id' => $user->id,
                'tenant_id' => $user->tenant_id,
                'count' => $importedCount,
            ]);
            
            return redirect()->route('tenant-admin.reviews.index')
                ->with('success', "Successfully imported {$importedCount} Google Business Profile locations.");
        } catch (\Exception $e) {
            Log::error('Failed to import locations: ' . $e->getMessage());
            return redirect()->route('tenant-admin.reviews.index')
                ->with('error', 'Failed to import locations: ' . $e->getMessage());
        }
    }

    /**
     * Ingest reviews - Tenant Admin can ingest reviews from their tenant's locations
     */
    public function ingest()
    {
        try {
            $locations = $this->userContext->getUserLocations()->get();
            
            if ($locations->count() === 0) {
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('error', 'No Google Business Profile locations found. Please import locations first.');
            }
            
            $ingestedCount = 0;
            $gbpService = app(GoogleBusinessProfile::class);
            
            foreach ($locations as $location) {
                $accountId = basename($location->account_id);
                $locationId = basename($location->location_id);
                $reviews = $gbpService->listReviews($accountId, $locationId);
                
                foreach ($reviews as $reviewData) {
                    Review::updateOrCreate(
                        [
                            'location_id' => $location->id,
                            'external_review_id' => $reviewData['id'],
                        ],
                        [
                            'user_id' => $location->user_id,
                            'tenant_id' => $location->tenant_id,
                            'source' => 'google_gbp',
                            'stars' => $reviewData['stars'],
                            'lang' => $reviewData['lang'] ?? 'en',
                            'text' => $reviewData['comment'],
                            'status' => 'new',
                        ]
                    );
                    $ingestedCount++;
                }
            }

            Log::info('Tenant reviews: ingested reviews', [
                'user_id' => auth()->id(),
                'tenant_id' => auth()->user()->tenant_id,
                'count' => $ingestedCount,
            ]);
            
            return redirect()->route('tenant-admin.reviews.index')
                ->with('success', "Successfully ingested {$ingestedCount} reviews.");
        } catch (\Exception $e) {
            Log::error('Failed to ingest reviews: ' . $e->getMessage());
            return redirect()->route('tenant-admin.reviews.index')
                ->with('error', 'Failed to ingest reviews: ' . $e->getMessage());
        }
    }

    /**
     * Generate AI draft for review - Tenant Admin can draft reviews within their tenant
     */
    public function draft(Review $review)
    {
        try {
            // Verify review belongs to tenant
            $userLocations = $this->userContext->getUserLocations();
            if (!$userLocations->where('id', $review->location_id)->exists()) {
                abort(403, 'You do not have permission to draft this review.');
            }

            $aiManager = app(AIServiceManager::class);
            $draftService = new EnhancedReplyDraftService($aiManager);
            $result = $draftService->makeForReview($review);

            if (!empty($result['meta']['error'])) {
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('error', 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error'));
            }
            
            $draft = ReviewDraft::create([
                'review_id' => $review->id,
                'user_id' => $review->user_id,
                'tenant_id' => $review->tenant_id,
                'draft_reply' => $result['draft'],
                'ai_meta' => $result['meta'],
                'status' => 'pending',
            ]);
            
            $review->update(['status' => 'drafted']);
            
            return redirect()->route('tenant-admin.reviews.index')
                ->with('success', 'AI draft generated successfully');
        } catch (\Exception $e) {
            Log::error('Failed to generate review draft: ' . $e->getMessage());
            return redirect()->route('tenant-admin.reviews.index')
                ->with('error', 'Failed to generate draft: ' . $e->getMessage());
        }
    }

    /**
     * Approve and post review reply - Tenant Admin can approve reviews within their tenant
     */
    public function approve(Review $review)
    {
        try {
            // Verify review belongs to tenant
            $userLocations = $this->userContext->getUserLocations();
            if (!$userLocations->where('id', $review->location_id)->exists()) {
                abort(403, 'You do not have permission to approve this review.');
            }
            
            $draft = $review->latestDraft();
            if (!$draft) {
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('error', 'No draft found for this review');
            }
            
            $gbpService = app(GoogleBusinessProfile::class);
            $result = $gbpService->replyToReview(
                $review->location->account_id,
                $review->location->location_id,
                $review->external_review_id,
                $draft->draft_reply
            );
            
            if ($result['success']) {
                $draft->update(['status' => 'sent']);
                $review->update([
                    'status' => 'posted',
                    'reviewed_at' => now(),
                ]);
                
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('success', 'Reply posted successfully');
            } else {
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('error', 'Failed to post reply: ' . $result['message']);
            }
        } catch (\Exception $e) {
            Log::error('Failed to post review reply: ' . $e->getMessage());
            return redirect()->route('tenant-admin.reviews.index')
                ->with('error', 'Failed to post reply: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete reviews - Tenant Admin can delete reviews within their tenant
     */
    public function bulkDelete(Request $request)
    {
        try {
            $reviewIds = $request->input('review_ids', []);
            
            if (empty($reviewIds)) {
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('error', 'No reviews selected for deletion.');
            }

            // Get tenant-scoped reviews
            $userLocations = $this->userContext->getUserLocations();
            $locationIds = $userLocations->pluck('id');
            
            $reviewsToDelete = Review::whereIn('location_id', $locationIds)
                ->whereIn('id', $reviewIds)
                ->get();

            if ($reviewsToDelete->count() !== count($reviewIds)) {
                return redirect()->route('tenant-admin.reviews.index')
                    ->with('error', 'Some selected reviews could not be found or you do not have permission to delete them.');
            }

            $deletedCount = 0;
            foreach ($reviewsToDelete as $review) {
                // Delete associated drafts first
                $review->drafts()->delete();
                
                // Delete the review
                $review->delete();
                $deletedCount++;
            }

            return redirect()->route('tenant-admin.reviews.index')
                ->with('success', "Successfully deleted {$deletedCount} review(s).");

        } catch (\Exception $e) {
            Log::error('Failed to bulk delete reviews: ' . $e->getMessage());
            return redirect()->route('tenant-admin.reviews.index')
                ->with('error', 'Failed to delete reviews: ' . $e->getMessage());
        }
    }

    /**
     * Delete individual review - Tenant Admin can only delete reviews within their tenant
     */
    public function destroy(Review $review)
    {
        try {
            // Verify review belongs to tenant
            $userLocations = $this->userContext->getUserLocations();
            if (!$userLocations->where('id', $review->location_id)->exists()) {
                abort(403, 'You do not have permission to delete this review.');
            }

            // Delete associated drafts first
            $review->drafts()->delete();
            
            // Delete the review
            $review->delete();

            return redirect()->route('tenant-admin.reviews.index')
                ->with('success', 'Review deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to delete review: ' . $e->getMessage());
            return redirect()->route('tenant-admin.reviews.index')
                ->with('error', 'Failed to delete review: ' . $e->getMessage());
        }
    }
}
