<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Location;
use App\Models\Review;
use App\Models\ReviewDraft;
use App\Models\OauthConnection;
use App\Services\GoogleBusinessProfile;
use App\Services\EnhancedReplyDraftService;
use App\Services\AIServiceManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReviewController extends Controller
{
    /**
     * Display reviews - Super Admin sees only their imported reviews
     */
    public function index()
    {
        try {
            // Super Admin sees only their own reviews
            $reviews = Review::where('user_id', auth()->id())
                ->with(['location', 'drafts', 'user', 'tenant'])
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            $googleConnectionsQuery = OauthConnection::where('user_id', auth()->id())
                ->where('provider', 'google');
            if (auth()->user()?->tenant_id) {
                $googleConnectionsQuery->where('tenant_id', auth()->user()->tenant_id);
            } else {
                $googleConnectionsQuery->whereNull('tenant_id');
            }
            $googleConnections = $googleConnectionsQuery->get();
            $aiManager = app(AIServiceManager::class);
            $providerInfo = $aiManager->getProviderInfo();
            $defaultKey = collect($providerInfo)->filter(fn ($info) => $info['is_default'])->keys()->first();
            $defaultProvider = $defaultKey ? $providerInfo[$defaultKey] : null;

            $locationsQuery = Location::where('user_id', auth()->id());
            if (auth()->user()?->tenant_id) {
                $locationsQuery->where('tenant_id', auth()->user()->tenant_id);
            } else {
                $locationsQuery->whereNull('tenant_id');
            }
            $locations = $locationsQuery->get();

            $status = [
                'oauth_connected' => $googleConnections->isNotEmpty(),
                'oauth_emails' => $googleConnections->pluck('account_email')->filter()->values()->all(),
                'locations_count' => $locations->count(),
                'reviews_count' => $reviews->total(),
                'ai_default_name' => $defaultProvider['name'] ?? 'Not set',
                'ai_default_ready' => $defaultProvider['configured'] ?? false,
            ];

            return view('super-admin.reviews.index', compact('reviews', 'status'));
        } catch (\Exception $e) {
            Log::error('Super Admin reviews index error: ' . $e->getMessage());
            return view('super-admin.reviews.index', ['reviews' => collect()]);
        }
    }

    public function disconnectGoogle()
    {
        $user = Auth::user();

        if (!$user) {
            return redirect()->route('super-admin.reviews.index')
                ->with('error', 'You must be logged in to disconnect Google Business Profile.');
        }

        DB::transaction(function () use ($user) {
            $reviewsQuery = Review::where('user_id', $user->id);
            if ($user->tenant_id) {
                $reviewsQuery->where('tenant_id', $user->tenant_id);
            } else {
                $reviewsQuery->whereNull('tenant_id');
            }

            $reviewIds = $reviewsQuery->pluck('id');
            ReviewDraft::whereIn('review_id', $reviewIds)->delete();
            $reviewsQuery->delete();

            $locationsQuery = Location::where('user_id', $user->id);
            if ($user->tenant_id) {
                $locationsQuery->where('tenant_id', $user->tenant_id);
            } else {
                $locationsQuery->whereNull('tenant_id');
            }
            $locationsQuery->delete();

            $oauthQuery = OauthConnection::where('provider', 'google')
                ->where('user_id', $user->id);
            if ($user->tenant_id) {
                $oauthQuery->where('tenant_id', $user->tenant_id);
            } else {
                $oauthQuery->whereNull('tenant_id');
            }
            $oauthQuery->delete();
        });

        return redirect()->route('super-admin.reviews.index')
            ->with('success', 'Disconnected Google Business Profile and removed associated data.');
    }

    /**
     * Import locations - Super Admin can import their own locations
     */
    public function importLocations()
    {
        try {
            $user = Auth::user();
            $importedCount = 0;
            
            $gbpService = app(GoogleBusinessProfile::class);
            $gbpLocations = $gbpService->listLocations();
            
            foreach ($gbpLocations as $gbpLocation) {
                Location::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'account_id' => $gbpLocation['account_id'],
                        'location_id' => $gbpLocation['location_id'],
                    ],
                    [
                        'source' => 'google_gbp',
                        'display_name' => $gbpLocation['display_name'],
                    ]
                );
                $importedCount++;
            }

            Log::info('Super admin reviews: imported locations', [
                'user_id' => $user->id,
                'tenant_id' => $user->tenant_id,
                'count' => $importedCount,
            ]);
            
            return redirect()->route('super-admin.reviews.index')
                ->with('success', "Successfully imported {$importedCount} Google Business Profile locations.");
        } catch (\Exception $e) {
            Log::error('Failed to import locations: ' . $e->getMessage());
            return redirect()->route('super-admin.reviews.index')
                ->with('error', 'Failed to import locations: ' . $e->getMessage());
        }
    }

    /**
     * Ingest reviews - Super Admin can ingest reviews from their own locations
     */
    public function ingest()
    {
        try {
            // Super Admin sees only their own locations
            $locations = Location::where('user_id', auth()->id())->get();
            
            if ($locations->count() === 0) {
                return redirect()->route('super-admin.reviews.index')
                    ->with('error', 'No Google Business Profile locations found. Please import locations first.');
            }
            
            $ingestedCount = 0;
            $gbpService = app(GoogleBusinessProfile::class);
            
            foreach ($locations as $location) {
                $accountId = basename($location->account_id);
                $locationId = basename($location->location_id);
                $reviews = $gbpService->listReviews($accountId, $locationId);
                
                foreach ($reviews as $reviewData) {
                    $starRating = $this->convertStarRatingToInt($reviewData['stars'] ?? null);
                    if ($starRating === null) {
                        Log::warning('Skipping review with invalid star rating', [
                            'review_id' => $reviewData['id'] ?? null,
                            'stars' => $reviewData['stars'] ?? null,
                        ]);
                        continue;
                    }

                    Review::updateOrCreate(
                        [
                            'location_id' => $location->id,
                            'external_review_id' => $reviewData['id'],
                        ],
                        [
                            'user_id' => $location->user_id,
                            'tenant_id' => $location->tenant_id,
                            'source' => 'google_gbp',
                            'stars' => $starRating,
                            'lang' => $reviewData['lang'] ?? 'en',
                            'text' => $reviewData['comment'],
                            'status' => 'new',
                        ]
                    );
                    $ingestedCount++;
                }
            }

            Log::info('Super admin reviews: ingested reviews', [
                'user_id' => auth()->id(),
                'tenant_id' => auth()->user()->tenant_id,
                'count' => $ingestedCount,
            ]);
            
            return redirect()->route('super-admin.reviews.index')
                ->with('success', "Successfully ingested {$ingestedCount} reviews.");
        } catch (\Exception $e) {
            Log::error('Failed to ingest reviews: ' . $e->getMessage());
            return redirect()->route('super-admin.reviews.index')
                ->with('error', 'Failed to ingest reviews: ' . $e->getMessage());
        }
    }

    /**
     * Convert text star ratings to integers
     *
     * @param mixed $stars
     * @return int|null
     */
    private function convertStarRatingToInt($stars)
    {
        if (is_numeric($stars)) {
            $intValue = (int) $stars;
            return ($intValue >= 1 && $intValue <= 5) ? $intValue : null;
        }

        if (is_string($stars)) {
            $stars = strtoupper(trim($stars));
            $mapping = [
                'ONE' => 1,
                'TWO' => 2,
                'THREE' => 3,
                'FOUR' => 4,
                'FIVE' => 5,
            ];
            return $mapping[$stars] ?? null;
        }

        return null;
    }

    /**
     * Generate AI draft for review - Super Admin can draft any review
     */
    public function draft(Review $review)
    {
        try {
            if ($review->user_id !== auth()->id()) {
                abort(403, 'You do not have permission to draft this review.');
            }

            $aiManager = app(AIServiceManager::class);
            $draftService = new EnhancedReplyDraftService($aiManager);
            $result = $draftService->makeForReview($review);

            if (!empty($result['meta']['error'])) {
                return redirect()->route('super-admin.reviews.index')
                    ->with('error', 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error'));
            }
            
            $draft = ReviewDraft::create([
                'review_id' => $review->id,
                'user_id' => $review->user_id,
                'tenant_id' => $review->tenant_id,
                'draft_reply' => $result['draft'],
                'ai_meta' => $result['meta'],
                'status' => 'pending',
            ]);
            
            $review->update(['status' => 'drafted']);
            
            return redirect()->route('super-admin.reviews.index')
                ->with('success', 'AI draft generated successfully');
        } catch (\Exception $e) {
            Log::error('Failed to generate review draft: ' . $e->getMessage());
            return redirect()->route('super-admin.reviews.index')
                ->with('error', 'Failed to generate draft: ' . $e->getMessage());
        }
    }

    /**
     * Approve and post review reply - Super Admin can approve any review
     */
    public function approve(Review $review)
    {
        try {
            if ($review->user_id !== auth()->id()) {
                abort(403, 'You do not have permission to approve this review.');
            }

            $draft = $review->latestDraft();
            if (!$draft) {
                return redirect()->route('super-admin.reviews.index')
                    ->with('error', 'No draft found for this review');
            }
            
            $gbpService = app(GoogleBusinessProfile::class);
            $result = $gbpService->replyToReview(
                $review->location->account_id,
                $review->location->location_id,
                $review->external_review_id,
                $draft->draft_reply
            );
            
            if ($result['success']) {
                $draft->update(['status' => 'sent']);
                $review->update([
                    'status' => 'posted',
                    'reviewed_at' => now(),
                ]);
                
                return redirect()->route('super-admin.reviews.index')
                    ->with('success', 'Reply posted successfully');
            } else {
                return redirect()->route('super-admin.reviews.index')
                    ->with('error', 'Failed to post reply: ' . $result['message']);
            }
        } catch (\Exception $e) {
            Log::error('Failed to post review reply: ' . $e->getMessage());
            return redirect()->route('super-admin.reviews.index')
                ->with('error', 'Failed to post reply: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete reviews - Super Admin can delete any reviews
     */
    public function bulkDelete(Request $request)
    {
        try {
            $reviewIds = $request->input('review_ids', []);
            
            if (empty($reviewIds)) {
                return redirect()->route('super-admin.reviews.index')
                    ->with('error', 'No reviews selected for deletion.');
            }

            // Super Admin can delete only their reviews
            $reviewsToDelete = Review::where('user_id', auth()->id())
                ->whereIn('id', $reviewIds)
                ->get();

            if ($reviewsToDelete->count() !== count($reviewIds)) {
                return redirect()->route('super-admin.reviews.index')
                    ->with('error', 'Some selected reviews could not be found.');
            }

            $deletedCount = 0;
            foreach ($reviewsToDelete as $review) {
                // Delete associated drafts first
                $review->drafts()->delete();
                
                // Delete the review
                $review->delete();
                $deletedCount++;
            }

            return redirect()->route('super-admin.reviews.index')
                ->with('success', "Successfully deleted {$deletedCount} review(s).");

        } catch (\Exception $e) {
            Log::error('Failed to bulk delete reviews: ' . $e->getMessage());
            return redirect()->route('super-admin.reviews.index')
                ->with('error', 'Failed to delete reviews: ' . $e->getMessage());
        }
    }

    /**
     * Delete individual review - Super Admin can delete any review
     */
    public function destroy(Review $review)
    {
        try {
            if ($review->user_id !== auth()->id()) {
                abort(403, 'You do not have permission to delete this review.');
            }

            // Delete associated drafts first
            $review->drafts()->delete();
            
            // Delete the review
            $review->delete();

            return redirect()->route('super-admin.reviews.index')
                ->with('success', 'Review deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to delete review: ' . $e->getMessage());
            return redirect()->route('super-admin.reviews.index')
                ->with('error', 'Failed to delete review: ' . $e->getMessage());
        }
    }
}
