<?php


namespace App\Http\Controllers;

use App\Models\Location;
use App\Models\Review;
use App\Models\ReviewDraft;
use App\Models\OauthConnection;
use App\Services\GoogleBusinessProfile;
use App\Services\EnhancedReplyDraftService;
use App\Services\AIServiceManager;
use App\Services\UserContextService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReviewController extends Controller
{
    private GoogleBusinessProfile $gbpService;
    private EnhancedReplyDraftService $draftService;
    private UserContextService $userContext;

    public function __construct(GoogleBusinessProfile $gbpService, AIServiceManager $aiManager, UserContextService $userContext)
    {
        $this->gbpService = $gbpService;
        $this->draftService = new EnhancedReplyDraftService($aiManager);
        $this->userContext = $userContext;
    }

    /**
     * Display reviews with latest draft status
     */
    public function index()
    {
        // Get user-scoped reviews through locations
        $userLocations = $this->userContext->getUserLocations()->get();
        $locationIds = $userLocations->pluck('id');

        // Query reviews directly with proper pagination
        $reviews = \App\Models\Review::whereIn('location_id', $locationIds)
            ->with(['location', 'drafts'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $googleConnectionsQuery = OauthConnection::where('user_id', auth()->id())
            ->where('provider', 'google');
        if (auth()->user()?->tenant_id) {
            $googleConnectionsQuery->where('tenant_id', auth()->user()->tenant_id);
        } else {
            $googleConnectionsQuery->whereNull('tenant_id');
        }
        $googleConnections = $googleConnectionsQuery->get();
        $aiManager = app(AIServiceManager::class);
        $providerInfo = $aiManager->getProviderInfo();
        $defaultKey = collect($providerInfo)->filter(fn ($info) => $info['is_default'])->keys()->first();
        $defaultProvider = $defaultKey ? $providerInfo[$defaultKey] : null;

        $status = [
            'oauth_connected' => $googleConnections->isNotEmpty(),
            'oauth_emails' => $googleConnections->pluck('account_email')->filter()->values()->all(),
            'locations_count' => $userLocations->count(),
            'reviews_count' => $reviews->total(),
            'ai_default_name' => $defaultProvider['name'] ?? 'Not set',
            'ai_default_ready' => $defaultProvider['configured'] ?? false,
        ];

        return view('reviews.index', compact('reviews', 'status'));
    }

    public function disconnectGoogle()
    {
        $user = Auth::user();

        if (!$user) {
            return redirect()->route('reviews.index')
                ->with('error', 'You must be logged in to disconnect Google Business Profile.');
        }

        DB::transaction(function () use ($user) {
            $reviewsQuery = Review::where('user_id', $user->id);
            if ($user->tenant_id) {
                $reviewsQuery->where('tenant_id', $user->tenant_id);
            } else {
                $reviewsQuery->whereNull('tenant_id');
            }

            $reviewIds = $reviewsQuery->pluck('id');
            ReviewDraft::whereIn('review_id', $reviewIds)->delete();
            $reviewsQuery->delete();

            $locationsQuery = Location::where('user_id', $user->id);
            if ($user->tenant_id) {
                $locationsQuery->where('tenant_id', $user->tenant_id);
            } else {
                $locationsQuery->whereNull('tenant_id');
            }
            $locationsQuery->delete();

            $oauthQuery = OauthConnection::where('provider', 'google')
                ->where('user_id', $user->id);
            if ($user->tenant_id) {
                $oauthQuery->where('tenant_id', $user->tenant_id);
            } else {
                $oauthQuery->whereNull('tenant_id');
            }
            $oauthQuery->delete();
        });

        return redirect()->route('reviews.index')
            ->with('success', 'Disconnected Google Business Profile and removed associated data.');
    }

    /**
     * Import Google Business Profile locations first
     */
    public function importLocations()
    {
        try {
            $user = auth()->user();
            $importedCount = 0;

            // Get Google Business Profile locations
            $gbpLocations = $this->gbpService->listLocations();
            
            foreach ($gbpLocations as $gbpLocation) {
                \App\Models\Location::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'account_id' => $gbpLocation['account_id'],
                        'location_id' => $gbpLocation['location_id'],
                    ],
                    [
                        'source' => 'google_gbp',
                        'display_name' => $gbpLocation['display_name'],
                    ]
                );
                $importedCount++;
            }

            Log::info('Reviews: imported locations', [
                'user_id' => $user->id,
                'tenant_id' => $user->tenant_id,
                'count' => $importedCount,
            ]);

            return redirect()->route('reviews.index')
                ->with('success', "Successfully imported {$importedCount} Google Business Profile locations.");
                
        } catch (\Exception $e) {
            Log::error('Failed to import locations: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to import locations: ' . $e->getMessage());
        }
    }

    /**
     * Manually fetch reviews from Google Business Profile
     */
    public function ingest()
    {
        try {
            // Get user-scoped locations
            $locations = $this->userContext->getUserLocations()->get();
            
            if ($locations->count() === 0) {
                return redirect()->route('reviews.index')
                    ->with('error', 'No Google Business Profile locations found. Please import locations first.');
            }
            
            $ingestedCount = 0;

            foreach ($locations as $location) {
                // Extract numeric IDs from resource names
                $accountId = basename($location->account_id); // Extract "106708635971562662717" from "accounts/106708635971562662717"
                $locationId = basename($location->location_id); // Extract "13117477215661797539" from "locations/13117477215661797539"
                
                Log::info('Fetching reviews for location', [
                    'location_name' => $location->display_name,
                    'account_id' => $accountId,
                    'location_id' => $locationId
                ]);
                
                $reviews = $this->gbpService->listReviews($accountId, $locationId);
                
                Log::info('Retrieved reviews from API', [
                    'location_name' => $location->display_name,
                    'review_count' => count($reviews)
                ]);
                
                foreach ($reviews as $reviewData) {
                    // Add safety checks for required fields
                    if (!isset($reviewData['id']) || !isset($reviewData['stars']) || !isset($reviewData['comment'])) {
                        Log::warning('Skipping review with missing required fields', ['review_data' => $reviewData]);
                        continue;
                    }
                    
                    // Convert text star ratings to integers
                    $starRating = $this->convertStarRatingToInt($reviewData['stars']);
                    if ($starRating === null) {
                        Log::warning('Skipping review with invalid star rating', [
                            'review_id' => $reviewData['id'],
                            'stars' => $reviewData['stars']
                        ]);
                        continue;
                    }
                    
                    Review::updateOrCreate(
                        [
                            'location_id' => $location->id,
                            'external_review_id' => $reviewData['id'],
                        ],
                        [
                            'user_id' => auth()->id(),
                            'tenant_id' => auth()->user()->tenant_id,
                            'source' => 'google_gbp',
                            'stars' => $starRating,
                            'lang' => $reviewData['lang'] ?? 'en', // Default to 'en' if not provided
                            'text' => $reviewData['comment'],
                            'status' => 'new',
                            'reviewed_at' => isset($reviewData['createTime']) ? 
                                \Carbon\Carbon::parse($reviewData['createTime']) : 
                                now(),
                        ]
                    );
                    $ingestedCount++;
                }
            }

            Log::info('Reviews: ingested reviews', [
                'user_id' => auth()->id(),
                'tenant_id' => auth()->user()->tenant_id,
                'count' => $ingestedCount,
            ]);

            return redirect()->route('reviews.index')
                ->with('success', "Successfully ingested {$ingestedCount} reviews.");
                
        } catch (\Exception $e) {
            Log::error('Failed to ingest reviews: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to ingest reviews: ' . $e->getMessage());
        }
    }

    /**
     * Generate AI draft for a review
     */
    public function draft(Review $review)
    {
        try {
            $result = $this->draftService->makeForReview($review);

            if (!empty($result['meta']['error'])) {
                return redirect()->route('reviews.index')
                    ->with('error', 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error'));
            }
            
            $draft = ReviewDraft::create([
                'user_id' => auth()->id(),
                'tenant_id' => auth()->user()->tenant_id,
                'review_id' => $review->id,
                'draft_reply' => $result['draft'],
                'ai_meta' => $result['meta'],
                'status' => 'pending',
            ]);
            
            $review->update(['status' => 'drafted']);
            
            return redirect()->route('reviews.index')
                ->with('success', 'AI draft generated successfully.');
                
        } catch (\Exception $e) {
            Log::error('Failed to generate review draft: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to generate draft: ' . $e->getMessage());
        }
    }

    /**
     * Approve a review draft (admin only)
     */
    public function approve(Review $review)
    {
        \Log::info('ReviewController::approve method called');
        
        if (!auth()->user()->isAdmin()) {
            abort(403, 'Unauthorized. Admin access required.');
        }
        
        $draft = $review->latestDraft();
        
        if (!$draft || $draft->status !== 'pending') {
            return redirect()->route('reviews.index')
                ->with('error', 'No pending draft found for this review.');
        }
        
        $draft->update([
            'status' => 'approved',
            'approved_by_user_id' => auth()->id(),
            'approved_at' => now(),
        ]);
        
        return redirect()->route('reviews.index')
            ->with('success', 'Draft approved successfully.');
    }

    /**
     * Post approved draft to Google Business Profile (admin only)
     */
    public function post(Review $review)
    {
        if (!auth()->user()->isAdmin()) {
            abort(403, 'Unauthorized. Admin access required.');
        }
        
        $draft = $review->latestDraft();
        
        if (!$draft || $draft->status !== 'approved') {
            return redirect()->route('reviews.index')
                ->with('error', 'No approved draft found for this review.');
        }
        
        try {
            $result = $this->gbpService->replyToReview(
                $review->location->account_id,
                $review->location->location_id,
                $review->external_review_id,
                $draft->draft_reply
            );
            
            if ($result['success']) {
                $draft->update(['status' => 'sent']);
                $review->update([
                    'status' => 'posted',
                    'reviewed_at' => now(),
                ]);
                
                return redirect()->route('reviews.index')
                    ->with('success', 'Reply posted successfully.');
            } else {
                return redirect()->route('reviews.index')
                    ->with('error', 'Failed to post reply: ' . $result['message']);
            }
            
        } catch (\Exception $e) {
            Log::error('Failed to post review reply: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to post reply: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing a review draft
     */
    public function edit(Review $review)
    {
        $draft = $review->latestDraft();
        
        if (!$draft) {
            return redirect()->route('reviews.index')
                ->with('error', 'No draft found for this review.');
        }
        
        return view('reviews.edit', compact('review', 'draft'));
    }

    /**
     * Update a review draft
     */
    public function update(Request $request, Review $review)
    {
        $request->validate([
            'draft_reply' => 'required|string|max:2000',
        ]);

        $draft = $review->latestDraft();
        
        if (!$draft) {
            return redirect()->route('reviews.index')
                ->with('error', 'No draft found for this review.');
        }

        $draft->update([
            'draft_reply' => $request->draft_reply,
        ]);

        return redirect()->route('reviews.index')
            ->with('success', 'Draft updated successfully.');
    }

    /**
     * Delete a review
     */
    public function deleteReview(Review $review)
    {
        try {
            $review->delete();
            
            return redirect()->route('reviews.index')
                ->with('success', 'Review deleted successfully.');
                
        } catch (\Exception $e) {
            Log::error('Failed to delete review: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to delete review: ' . $e->getMessage());
        }
    }

    /**
     * Delete a review draft
     */
    public function deleteDraft(ReviewDraft $reviewDraft)
    {
        try {
            $review = $reviewDraft->review;
            $reviewDraft->delete();

            if ($review && $review->status === 'drafted' && ! $review->drafts()->exists()) {
                $review->update(['status' => 'new']);
            }
            
            return redirect()->route('reviews.index')
                ->with('success', 'Draft deleted successfully.');
                
        } catch (\Exception $e) {
            Log::error('Failed to delete review draft: ' . $e->getMessage());
            return redirect()->route('reviews.index')
                ->with('error', 'Failed to delete draft: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete reviews
     */
    public function bulkDelete(Request $request)
    {
        if (!auth()->user()->isAdmin()) {
            abort(403, 'Unauthorized. Admin access required.');
        }
        
        $reviewIds = $request->input('review_ids', []);
        
        if (empty($reviewIds)) {
            return redirect()->route('reviews.index')
                ->with('error', 'No reviews selected for deletion.');
        }
        
        // Get user-scoped reviews to ensure we only delete reviews the user has access to
        $userLocations = $this->userContext->getUserLocations()->get();
        $locationIds = $userLocations->pluck('id');
        
        $deletedCount = Review::whereIn('id', $reviewIds)
            ->whereIn('location_id', $locationIds)
            ->count();
            
        Review::whereIn('id', $reviewIds)
            ->whereIn('location_id', $locationIds)
            ->delete();
        
        return redirect()->route('reviews.index')
            ->with('success', "Successfully deleted {$deletedCount} review(s) and all associated data.");
    }

    /**
     * Generate AI draft using enhanced multi-provider service
     * This is an example method showing how to use the new AI system
     */
    public function draftEnhanced(Review $review, Request $request)
    {
        try {
            $aiManager = app(AIServiceManager::class);
            $preferredProvider = $request->input('ai_provider'); // Optional provider override
            
            $enhancedService = new EnhancedReplyDraftService($aiManager, $preferredProvider);
            $result = $enhancedService->makeForReview($review);

            if (!empty($result['meta']['error'])) {
                return redirect()->route('reviews.index')
                    ->with('error', 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error'));
            }
            
            // Create draft with AI provider info
            $draft = ReviewDraft::create([
                'review_id' => $review->id,
                'draft_reply' => $result['draft'],
                'status' => 'pending',
                'ai_provider' => $result['meta']['ai_provider'] ?? 'unknown',
                'meta' => json_encode($result['meta']),
            ]);
            
            $review->update(['status' => 'drafted']);
            
            return response()->json([
                'success' => true,
                'draft' => $result['draft'],
                'provider' => $result['meta']['ai_provider'] ?? 'unknown',
                'meta' => $result['meta'],
            ]);
            
        } catch (\Exception $e) {
            Log::error('Enhanced AI draft generation failed: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    
    /**
     * Convert text star ratings to integers
     * 
     * @param mixed $stars
     * @return int|null
     */
    private function convertStarRatingToInt($stars)
    {
        // Handle integer values
        if (is_numeric($stars)) {
            $intValue = (int) $stars;
            return ($intValue >= 1 && $intValue <= 5) ? $intValue : null;
        }
        
        // Handle text values
        if (is_string($stars)) {
            $stars = strtoupper(trim($stars));
            
            $mapping = [
                'ONE' => 1,
                'TWO' => 2,
                'THREE' => 3,
                'FOUR' => 4,
                'FIVE' => 5,
                '1' => 1,
                '2' => 2,
                '3' => 3,
                '4' => 4,
                '5' => 5,
            ];
            
            return $mapping[$stars] ?? null;
        }
        
        return null;
    }
}
