<?php

namespace App\Http\Controllers;

use App\Models\EmailDraft;
use App\Models\EmailSignature;
use App\Models\MailAccount;
use App\Models\Message;
use App\Models\Thread;
use App\Services\GmailService;
use App\Services\EnhancedReplyDraftService;
use App\Services\AIServiceManager;
use App\Services\SMTPEmailService;
use App\Services\IMAPEmailService;
use App\Services\UserContextService;
use App\Models\EmailProvider;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class EmailController extends Controller
{
    use AuthorizesRequests;

    private GmailService $gmailService;
    private EnhancedReplyDraftService $draftService;
    private SMTPEmailService $smtpService;
    private IMAPEmailService $imapService;
    private UserContextService $userContext;

    public function __construct(GmailService $gmailService, AIServiceManager $aiManager, SMTPEmailService $smtpService, IMAPEmailService $imapService, UserContextService $userContext)
    {
        $this->gmailService = $gmailService;
        $this->draftService = new EnhancedReplyDraftService($aiManager);
        $this->smtpService = $smtpService;
        $this->imapService = $imapService;
        $this->userContext = $userContext;
    }

    /**
     * Display email threads
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        // Super admin inbox should be limited to their own threads
        if ($user && $user->isSuperAdmin()) {
            $query = Thread::where('user_id', $user->id);
        } else {
            // Get user-scoped threads
            $query = $this->userContext->getUserThreads();
        }
        
        // Filter by mail account if specified
        if ($request->has('account') && $request->account !== 'all') {
            $query = $query->where('mail_account_id', $request->account);
        }
        
        $threads = $query->orderBy('last_message_at', 'desc')->paginate(20);
        
        // Get user-scoped mail accounts for the filter dropdown
        if ($user && $user->isSuperAdmin()) {
            $mailAccounts = MailAccount::where('user_id', $user->id)->get();
        } else {
            $mailAccounts = $this->userContext->getUserMailAccounts()->get();
        }
        
        return view('email.index', compact('threads', 'mailAccounts'));
    }

    /**
     * Manually fetch emails from Gmail OAuth or IMAP accounts
     */
    public function ingest(Request $request)
    {
        try {
            $accountId = $request->get('account_id');
            
            // Debug logging
            Log::info('Email ingest request', [
                'account_id' => $accountId,
                'all_request_data' => $request->all()
            ]);
            
            $user = auth()->user();
            if ($accountId) {
                if ($user->isSuperAdmin()) {
                    $mailAccount = MailAccount::where('user_id', $user->id)
                        ->where('id', $accountId)
                        ->first();
                } else {
                    // Find account within user's scope
                    $mailAccount = $this->userContext->getUserMailAccounts()->where('id', $accountId)->first();
                }
                Log::info('Found mail account by ID', ['account' => $mailAccount ? $mailAccount->email : 'null']);
            } else {
                if ($user->isSuperAdmin()) {
                    $mailAccount = MailAccount::where('user_id', $user->id)
                        ->where('is_primary', true)
                        ->first();
                } else {
                    // Find primary account within user's scope
                    $mailAccount = $this->userContext->getUserMailAccounts()->where('is_primary', true)->first();
                }
                Log::info('Using primary mail account', ['account' => $mailAccount ? $mailAccount->email : 'null']);
            }
            
            if (!$mailAccount) {
                Log::error('No mail account found for ingest');
                return redirect()->route('email.index')
                    ->with('error', 'No mail account found. Please connect your email account first.');
            }
            
            Log::info('Starting email ingest for account', [
                'email' => $mailAccount->email,
                'connection_type' => $mailAccount->connection_type
            ]);

            // Route to appropriate service based on account type
            if ($mailAccount->isOAuth()) {
                return $this->ingestOAuthEmails($mailAccount);
            } elseif ($mailAccount->hasImapConfig()) {
                return $this->ingestImapEmails($mailAccount);
            } else {
                return redirect()->route('email.index')
                    ->with('error', 'Email account not configured for fetching. Please configure OAuth or IMAP settings.');
            }
                
        } catch (\Exception $e) {
            Log::error('Failed to ingest emails: ' . $e->getMessage());
            return redirect()->route('email.index')
                ->with('error', 'Failed to ingest emails: ' . $e->getMessage());
        }
    }

    /**
     * Fetch emails from OAuth account (Gmail)
     */
    private function ingestOAuthEmails(MailAccount $mailAccount): \Illuminate\Http\RedirectResponse
    {
        try {
            $startTime = microtime(true);
            $timeLimitSeconds = 8;
            $maxResults = 10;
            $messageIds = $this->gmailService->listRecentMessageIds($mailAccount, 'newer_than:1d -from:me in:inbox', $maxResults);
            $ingestedCount = 0;

            foreach ($messageIds as $messageId) {
                if ((microtime(true) - $startTime) > $timeLimitSeconds) {
                    Log::info('Email ingest: Stopping early to avoid request timeout', [
                        'email' => $mailAccount->email,
                        'ingested_count' => $ingestedCount,
                    ]);
                    break;
                }

                $messageData = $this->gmailService->getMessageFull($mailAccount, $messageId);
                
                if (empty($messageData)) {
                    continue;
                }

                $headers = $messageData['headers'];
                $threadId = $messageData['threadId'];

                // Create or update thread
                $user = auth()->user();
                $thread = Thread::updateOrCreate(
                    ['provider_thread_id' => $threadId],
                    [
                        'mail_account_id' => $mailAccount->id,
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'subject' => $headers['Subject'] ?? 'No Subject',
                        'participants' => $this->extractParticipants($headers),
                        'last_message_at' => now(),
                        'status' => 'open',
                    ]
                );
                
                // Always update last_message_at to current time for fetched emails
                $thread->update(['last_message_at' => now()]);

                // Create or update message
                $user = auth()->user();
                Message::updateOrCreate(
                    ['provider_msg_id' => $messageData['id']],
                    [
                        'thread_id' => $thread->id,
                        'direction' => 'inbound',
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'from_addr' => $headers['From'] ?? '',
                        'to_json' => $this->parseEmailAddresses($headers['To'] ?? ''),
                        'cc_json' => $this->parseEmailAddresses($headers['Cc'] ?? ''),
                        'bcc_json' => $this->parseEmailAddresses($headers['Bcc'] ?? ''),
                        'date' => $headers['Date'] ?? now(),
                        'snippet' => $messageData['snippet'],
                        'body_text' => $messageData['body_text'],
                        'body_html' => $messageData['body_html'],
                    ]
                );

                $ingestedCount++;
            }

            return redirect()->route('email.index')
                ->with('success', "Successfully ingested {$ingestedCount} Gmail messages from {$mailAccount->email}.");

        } catch (\Exception $e) {
            Log::error('Failed to ingest OAuth emails: ' . $e->getMessage());
            return redirect()->route('email.index')
                ->with('error', 'Failed to fetch Gmail emails: ' . $e->getMessage());
        }
    }

    /**
     * Fetch emails from IMAP account
     */
    private function ingestImapEmails(MailAccount $mailAccount): \Illuminate\Http\RedirectResponse
    {
        try {
            $result = $this->imapService->fetchEmails($mailAccount, 20); // Fetch last 20 emails

            if (!$result['success']) {
                return redirect()->route('email.index')
                    ->with('error', $result['message']);
            }

            if (empty($result['emails'])) {
                return redirect()->route('email.index')
                    ->with('success', "No new emails found in {$mailAccount->email}.");
            }

            $processedCount = $this->processImapEmails($result['emails'], $mailAccount);

            return redirect()->route('email.index')
                ->with('success', "Successfully processed {$processedCount} IMAP messages from {$mailAccount->email}.");

        } catch (\Exception $e) {
            Log::error('Failed to ingest IMAP emails: ' . $e->getMessage());
            return redirect()->route('email.index')
                ->with('error', 'Failed to fetch IMAP emails: ' . $e->getMessage());
        }
    }

    /**
     * Generate AI draft for a thread
     */
    public function draft(Thread $thread)
    {
        try {
            $lastInboundMessage = $thread->latestInboundMessage();
            
            if (!$lastInboundMessage) {
                return redirect()->route('email.index')
                    ->with('error', 'No inbound message found for this thread.');
            }

            $lastInboundMessage = $this->refreshMessageBodyIfNeeded($thread, $lastInboundMessage);
            $result = $this->draftService->makeForEmail($thread, $lastInboundMessage);

            if (!empty($result['meta']['error'])) {
                return redirect()->route('email.index')
                    ->with('error', 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error'));
            }
            
            EmailDraft::create([
                'user_id' => auth()->id(),
                'tenant_id' => auth()->user()->tenant_id,
                'thread_id' => $thread->id,
                'in_reply_to_message_id' => $lastInboundMessage->id,
                'subject' => 'Re: ' . $thread->subject,
                'draft_html' => $result['draft'],
                'ai_meta' => $result['meta'],
                'status' => 'pending',
            ]);
            
            return redirect()->route('email.index')
                ->with('success', 'AI draft generated successfully.');
                
        } catch (\Exception $e) {
            Log::error('Failed to generate email draft: ' . $e->getMessage());
            return redirect()->route('email.index')
                ->with('error', 'Failed to generate draft: ' . $e->getMessage());
        }
    }

    private function refreshMessageBodyIfNeeded(Thread $thread, Message $message): Message
    {
        $bodyText = trim((string) ($message->body_text ?? ''));
        $bodyHtml = trim((string) ($message->body_html ?? ''));
        $textLength = strlen($bodyText);
        $htmlTextLength = strlen(strip_tags($bodyHtml));

        if ($textLength >= 120 || $htmlTextLength >= 120) {
            return $message;
        }

        $mailAccount = $thread->mailAccount;
        if (!$mailAccount || !$mailAccount->isOAuth()) {
            return $message;
        }

        if (empty($message->provider_msg_id)) {
            return $message;
        }

        $messageData = $this->gmailService->getMessageFull($mailAccount, $message->provider_msg_id);
        if (empty($messageData)) {
            return $message;
        }

        $updatedBodyText = $messageData['body_text'] ?? '';
        $updatedBodyHtml = $messageData['body_html'] ?? '';
        $updatedSnippet = $messageData['snippet'] ?? $message->snippet;

        if (empty($updatedBodyText) && !empty($updatedBodyHtml)) {
            $updatedBodyText = trim(html_entity_decode(strip_tags($updatedBodyHtml), ENT_QUOTES | ENT_HTML5, 'UTF-8'));
        }

        if (empty($updatedSnippet) && !empty($updatedBodyText)) {
            $updatedSnippet = substr($updatedBodyText, 0, 250);
        }

        $message->update([
            'body_text' => $updatedBodyText,
            'body_html' => $updatedBodyHtml,
            'snippet' => $updatedSnippet,
        ]);

        return $message->fresh();
    }

    /**
     * Show edit form for email draft
     */
    public function edit(EmailDraft $draft)
    {
        // Get the mail account for this draft
        $mailAccount = $draft->thread->mailAccount;
        
        if (!$mailAccount) {
            return redirect()->route('email.index')
                ->with('error', 'Mail account not found for this draft.');
        }
        
        // Get available signatures for this account
        $signatures = $mailAccount->emailSignatures()->active()->get();
        
        return view('email.edit', compact('draft', 'mailAccount', 'signatures'));
    }

    /**
     * Update email draft
     */
    public function update(Request $request, EmailDraft $draft)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'draft_html' => 'required|string',
        ]);

        $draft->update([
            'subject' => $request->subject,
            'draft_html' => $request->draft_html,
        ]);

        return redirect()->route('email.index')
            ->with('success', 'Draft updated successfully.');
    }

    /**
     * Regenerate draft with AI (AJAX)
     */
    public function regenerate(Request $request, EmailDraft $draft)
    {
        try {
            $this->authorize('admin', auth()->user());
            
            $aiProvider = $request->input('ai_provider', 'gemini');
            $thread = $draft->thread;
            $lastInboundMessage = $thread->latestInboundMessage();
            $lastInboundMessage = $this->refreshMessageBodyIfNeeded($thread, $lastInboundMessage);
            
            if (!$lastInboundMessage) {
                return response()->json([
                    'success' => false,
                    'error' => 'No inbound message found for this thread.'
                ], 400);
            }

            // Use enhanced AI service if available
            if (class_exists('App\Services\EnhancedReplyDraftService')) {
                $aiManager = app('App\Services\AIServiceManager');
                $enhancedService = new \App\Services\EnhancedReplyDraftService($aiManager, $aiProvider);
                $result = $enhancedService->makeForEmail($thread, $lastInboundMessage);
            } else {
                $result = $this->draftService->makeForEmail($thread, $lastInboundMessage);
            }

            if (!empty($result['meta']['error'])) {
                return response()->json([
                    'success' => false,
                    'error' => 'AI draft failed: ' . ($result['meta']['message'] ?? 'Provider error')
                ], 400);
            }

            // Update the draft with new content
            $draft->update([
                'draft_html' => $result['draft'],
                'ai_meta' => $result['meta'],
                'ai_regeneration_count' => $draft->ai_regeneration_count + 1,
            ]);

            return response()->json([
                'success' => true,
                'draft' => $result['draft'],
                'provider' => $result['meta']['ai_provider'] ?? $aiProvider,
                'meta' => $result['meta'],
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to regenerate email draft: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to regenerate draft: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Approve and send email draft (admin only)
     */
    public function approveAndSend(EmailDraft $draft)
    {
        $this->authorize('admin', auth()->user());
        
        if ($draft->status !== 'pending') {
            return redirect()->route('email.index')
                ->with('error', 'Draft is not in pending status.');
        }
        
        try {
            $thread = $draft->thread;
            $mailAccount = $thread->mailAccount;
            
            // Add signature if available
            $html = $draft->draft_html;
            $defaultSignature = $mailAccount->emailSignatures()->default()->active()->first();
            if ($defaultSignature) {
                $html .= '<br><br>' . $defaultSignature->signature_html;
            }
            
            // Choose sending method based on account type
            if ($mailAccount->isOAuth()) {
                $result = $this->gmailService->sendReply(
                    $mailAccount,
                    $thread->provider_thread_id,
                    $draft->inReplyToMessage->from_addr,
                    $draft->subject,
                    $html,
                    $draft->inReplyToMessage->provider_msg_id
                );
            } else {
                // Use SMTP for non-OAuth accounts
                $result = $this->smtpService->sendEmail($mailAccount, [
                    'to' => $draft->inReplyToMessage->from_addr,
                    'subject' => $draft->subject,
                    'body' => $html,
                    'thread_id' => $thread->provider_thread_id,
                    'in_reply_to' => $draft->inReplyToMessage->provider_msg_id,
                ]);
                
                // Generate a message ID for SMTP emails
                $result['messageId'] = 'smtp_' . uniqid() . '@' . $mailAccount->email;
            }
            
            if ($result['success']) {
                // Create outbound message record
                Message::create([
                    'thread_id' => $thread->id,
                    'provider_msg_id' => $result['messageId'],
                    'direction' => 'outbound',
                    'from_addr' => $mailAccount->email,
                    'to_json' => [$draft->inReplyToMessage->from_addr],
                    'date' => now(),
                    'snippet' => strip_tags($html),
                    'body_text' => strip_tags($html),
                    'body_html' => $html,
                ]);
                
                $draft->update([
                    'status' => 'sent',
                    'approved_by_user_id' => auth()->id(),
                    'approved_at' => now(),
                ]);
                
                return redirect()->route('email.index')
                    ->with('success', 'Email sent successfully.');
            } else {
                return redirect()->route('email.index')
                    ->with('error', 'Failed to send email: ' . $result['message']);
            }
            
        } catch (\Exception $e) {
            Log::error('Failed to send email: ' . $e->getMessage());
            return redirect()->route('email.index')
                ->with('error', 'Failed to send email: ' . $e->getMessage());
        }
    }

    /**
     * Delete email draft (admin only)
     */
    public function destroy(EmailDraft $draft)
    {
        if (!auth()->user()->isAdmin() && $draft->user_id !== auth()->id()) {
            abort(403, 'This action is unauthorized.');
        }
        
        $draft->delete();
        
        return redirect()->route('email.index')
            ->with('success', 'Draft deleted successfully.');
    }

    /**
     * Delete entire email thread and all associated data (admin only)
     */
    public function destroyThread(Thread $thread)
    {
        if (!auth()->user()->isAdmin() && $thread->user_id !== auth()->id()) {
            abort(403, 'This action is unauthorized.');
        }
        
        // Delete all associated messages and drafts (cascade delete)
        $thread->delete();
        
        return redirect()->route('email.index')
            ->with('success', 'Email thread and all associated data deleted successfully.');
    }

    /**
     * Show email accounts management page
     */
    public function accounts()
    {
        // Get user-scoped mail accounts
        $mailAccounts = $this->userContext->getUserMailAccounts()->get();
        $emailProviders = EmailProvider::active()->get(); // Add SMTP providers
        
        return view('email.accounts', compact('mailAccounts', 'emailProviders'));
    }

    /**
     * Set primary email account
     */
    public function setPrimary(MailAccount $mailAccount)
    {
        $this->authorize('admin', auth()->user());
        
        // Ensure user can only set their own accounts as primary
        $userMailAccounts = $this->userContext->getUserMailAccounts()->get();
        if (!$userMailAccounts->contains('id', $mailAccount->id)) {
            return redirect()->route('email.accounts')
                ->with('error', 'You can only set your own email accounts as primary.');
        }
        
        // Remove primary from all user's accounts
        $userMailAccounts->where('is_primary', true)->each(function($account) {
            $account->update(['is_primary' => false]);
        });
        
        // Set this account as primary
        $mailAccount->update(['is_primary' => true]);
        
        return redirect()->route('email.accounts')
            ->with('success', 'Primary email account updated successfully.');
    }

    /**
     * Delete email account
     */
    public function deleteAccount(MailAccount $mailAccount)
    {
        $this->authorize('admin', auth()->user());
        
        // Ensure user can only delete their own accounts
        $userMailAccounts = $this->userContext->getUserMailAccounts()->get();
        if (!$userMailAccounts->contains('id', $mailAccount->id)) {
            return redirect()->route('email.accounts')
                ->with('error', 'You can only delete your own email accounts.');
        }
        
        // If this is the primary account, set another as primary
        if ($mailAccount->is_primary) {
            $newPrimary = $userMailAccounts->where('id', '!=', $mailAccount->id)->first();
            if ($newPrimary) {
                $newPrimary->update(['is_primary' => true]);
            }
        }
        
        $mailAccount->delete();
        
        return redirect()->route('email.accounts')
            ->with('success', 'Email account deleted successfully.');
    }

    /**
     * Bulk delete email threads
     */
    public function bulkDelete(Request $request)
    {
        $this->authorize('admin', auth()->user());
        
        $threadIds = $request->input('thread_ids', []);
        
        if (empty($threadIds)) {
            return redirect()->route('email.index')
                ->with('error', 'No threads selected for deletion.');
        }
        
        $deletedCount = Thread::whereIn('id', $threadIds)->count();
        Thread::whereIn('id', $threadIds)->delete();
        
        return redirect()->route('email.index')
            ->with('success', "Successfully deleted {$deletedCount} email thread(s) and all associated data.");
    }

    /**
     * Extract participants from email headers
     */
    private function extractParticipants(array $headers): array
    {
        $participants = [];
        
        if (isset($headers['From'])) {
            $participants[] = $headers['From'];
        }
        
        if (isset($headers['To'])) {
            $participants = array_merge($participants, $this->parseEmailAddresses($headers['To']));
        }
        
        return array_unique($participants);
    }

    /**
     * Parse email addresses from string
     */
    private function parseEmailAddresses(string $addresses): array
    {
        if (empty($addresses)) {
            return [];
        }
        
        // Simple parsing - split by comma and clean up
        $emails = array_map('trim', explode(',', $addresses));
        return array_filter($emails);
    }


    /**
     * Test SMTP connection before saving
     */
    public function testSmtpConnection(Request $request)
    {
        $request->validate([
            'smtp_host' => 'required|string',
            'smtp_port' => 'required|integer',
            'smtp_encryption' => 'required|string',
            'smtp_username' => 'required|string',
            'smtp_password' => 'required|string',
        ]);

        $smtpConfig = [
            'host' => $request->smtp_host,
            'port' => $request->smtp_port,
            'encryption' => $request->smtp_encryption,
            'username' => $request->smtp_username,
            'password' => $request->smtp_password,
        ];

        $result = $this->smtpService->testConnection($smtpConfig);

        return response()->json($result);
    }

    /**
     * Store SMTP account with IMAP configuration
     */
    public function storeSmtpWithImapAccount(Request $request)
    {
        try {
            Log::info('SMTP+IMAP account creation request', [
                'email' => $request->email,
                'has_imap' => $request->has('enable_imap'),
                'all_data' => $request->except(['smtp_password', 'imap_password']) // Don't log passwords
            ]);

            // All authenticated users (including agents) can create their own email accounts
            // No authorization check needed - users can only create accounts for themselves

            // Handle checkbox properly - convert "on" to true, absence to false
            $request->merge([
                'enable_imap' => $request->has('enable_imap') ? true : false
            ]);

            $request->validate([
                'email' => 'required|email',
                'from_name' => 'required|string',
                'smtp_host' => 'required|string',
                'smtp_port' => 'required|integer',
                'smtp_encryption' => 'required|string|in:none,tls,ssl',
                'smtp_username' => 'required|string',
                'smtp_password' => 'required|string',
                'smtp_from_name' => 'nullable|string',
                'provider_id' => 'nullable|string|exists:email_providers,id',
                
                // IMAP fields (optional)
                'enable_imap' => 'boolean',
                'imap_host' => 'required_if:enable_imap,true|nullable|string',
                'imap_port' => 'required_if:enable_imap,true|nullable|integer',
                'imap_encryption' => 'required_if:enable_imap,true|nullable|string|in:none,tls,ssl',
                'imap_username' => 'required_if:enable_imap,true|nullable|string',
                'imap_password' => 'required_if:enable_imap,true|nullable|string',
                'imap_folder' => 'nullable|string',
            ]);

            // Auto-fill IMAP settings if provider is selected and IMAP is enabled
            if ($request->filled('provider_id') && $request->has('enable_imap')) {
                $provider = EmailProvider::find($request->provider_id);
                if ($provider && $provider->imap_settings) {
                    $imapSettings = json_decode($provider->imap_settings, true);
                    $request->merge([
                        'imap_host' => $request->imap_host ?: $this->replaceDomainPlaceholders($imapSettings['host'], $request->email),
                        'imap_port' => $request->imap_port ?: ($imapSettings['port'] ?? 993),
                        'imap_encryption' => $request->imap_encryption ?: ($imapSettings['encryption'] ?? 'ssl'),
                    ]);
                }
            }

            Log::info('Creating SMTP+IMAP mail account', ['email' => $request->email]);

            // Auto-resolve IMAP settings if not provided
            if ($request->has('enable_imap') && empty($request->imap_host)) {
                $imapHost = $this->imapService->resolveImapHost($request->email);
                $request->merge([
                    'imap_host' => $imapHost,
                    'imap_port' => $request->imap_port ?: 993,
                    'imap_encryption' => $request->imap_encryption ?: 'ssl',
                    'imap_username' => $request->imap_username ?: $request->email,
                ]);
                
                Log::info('Auto-resolved IMAP settings', [
                    'email' => $request->email,
                    'imap_host' => $imapHost
                ]);
            }

            $user = auth()->user();
            $mailAccount = MailAccount::create([
                'email' => $request->email,
                'from_name' => $request->from_name,
                'connection_type' => 'smtp',
                'signature_html' => null,
                'user_id' => $user->id,
                'tenant_id' => $user->tenant_id,
                'smtp_host' => $request->smtp_host,
                'smtp_port' => $request->smtp_port,
                'smtp_encryption' => $request->smtp_encryption,
                'smtp_username' => $request->smtp_username,
                'smtp_password' => $request->smtp_password,
                'smtp_from_name' => $request->smtp_from_name,
                
                // IMAP configuration
                'imap_host' => $request->enable_imap ? $request->imap_host : null,
                'imap_port' => $request->enable_imap ? $request->imap_port : null,
                'imap_encryption' => $request->enable_imap ? $request->imap_encryption : null,
                'imap_username' => $request->enable_imap ? $request->imap_username : null,
                'imap_password' => $request->enable_imap ? $request->imap_password : null,
                'imap_folder' => $request->enable_imap ? ($request->imap_folder ?: 'INBOX') : 'INBOX',
            ]);

            // If this is the first account or marked as primary, set it as primary
            if ($request->has('make_primary') || MailAccount::count() === 1) {
                MailAccount::where('id', '!=', $mailAccount->id)->update(['is_primary' => false]);
                $mailAccount->update(['is_primary' => true]);
            }

            return redirect()->route('email.accounts')
                ->with('success', 'SMTP+IMAP account added successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to create SMTP+IMAP account: ' . $e->getMessage());
            return redirect()->route('email.accounts')
                ->with('error', 'Failed to create SMTP+IMAP account: ' . $e->getMessage());
        }
    }

    /**
     * Test IMAP connection before saving
     */
    public function testImapConnection(Request $request)
    {
        $request->validate([
            'imap_host' => 'required|string',
            'imap_port' => 'required|integer',
            'imap_encryption' => 'required|string',
            'imap_username' => 'required|string',
            'imap_password' => 'required|string',
            'imap_folder' => 'nullable|string',
        ]);

        try {
            // Create a temporary account object for testing
            $tempAccount = new MailAccount([
                'imap_host' => $request->imap_host,
                'imap_port' => $request->imap_port,
                'imap_encryption' => $request->imap_encryption,
                'imap_username' => $request->imap_username,
                'imap_password' => $request->imap_password,
                'imap_folder' => $request->imap_folder ?: 'INBOX',
            ]);

            // Use the simple IMAP test service
            $simpleTest = new \App\Services\SimpleIMAPTest();
            $result = $simpleTest->testConnection($tempAccount);
            
            Log::info('IMAP connection test result', [
                'account' => $tempAccount->email,
                'host' => $request->imap_host,
                'success' => $result['success']
            ]);

            return response()->json($result, 200, [], JSON_UNESCAPED_SLASHES);
            
        } catch (\Exception $e) {
            Log::error('IMAP test connection error: ' . $e->getMessage(), [
                'exception' => $e,
                'request_data' => $request->except(['imap_password'])
            ]);
            
            return response()->json([
                'success' => false,
                'message' => '❌ IMAP test failed: ' . $e->getMessage()
            ], 500, [], JSON_UNESCAPED_SLASHES);
        }
    }


    /**
     * Process and store IMAP emails into the database
     */
    private function processImapEmails(array $emails, MailAccount $mailAccount): int
    {// Parse email body and save to database
        $count = 0;

        foreach ($emails as $emailData) {
            try {
                // Generate thread ID (use subject + recipients as thread identifier)
                $threadId = $this->generateThreadId($emailData);

                // Find or create thread
                $thread = Thread::where('provider_thread_id', $threadId)
                    ->where('mail_account_id', $mailAccount->id)
                    ->first();

                if (!$thread) {
                    $user = auth()->user();
                    $thread = Thread::create([
                        'mail_account_id' => $mailAccount->id,
                        'provider_thread_id' => $threadId,
                        'subject' => $emailData['subject'],
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'participants' => json_encode([
                            'from' => $emailData['from_addr'],
                            'to' => $emailData['to_json'],
                            'cc' => $emailData['cc_json'] ?? []
                        ]),
                        'last_message_at' => now(), // Use current time so fetched emails appear at top
                        'status' => 'open'
                    ]);
                } else {
                    // Update existing thread's last_message_at to current time
                    $thread->update(['last_message_at' => now()]);
                }

                // Check if message already exists
                $existingMessage = Message::where('provider_msg_id', $emailData['provider_msg_id'])
                    ->where('thread_id', $thread->id)
                    ->first();

                if (!$existingMessage) {
                    $user = auth()->user();
                    Message::create([
                        'thread_id' => $thread->id,
                        'provider_msg_id' => $emailData['provider_msg_id'],
                        'direction' => 'inbound',
                        'user_id' => $user->id,
                        'tenant_id' => $user->tenant_id,
                        'from_addr' => $emailData['from_addr'],
                        'to_json' => $emailData['to_json'],
                        'cc_json' => $emailData['cc_json'] ?? [],
                        'date' => $emailData['date'],
                        'snippet' => $emailData['snippet'],
                        'body_text' => $emailData['body_text'],
                        'body_html' => $emailData['body_html'],
                        'attachments' => json_encode($emailData['attachments'] ?? [])
                    ]);

                    $count++;
                    
                    // Update thread last message date
                    $thread->update(['last_message_at' => $emailData['date']]);
                }

            } catch (\Exception $e) {
                Log::warning('Failed to process IMAP email', [
                    'error' => $e->getMessage(),
                    'email_data' => $emailData
                ]);
                continue;
            }
        }

        return $count;
    }

    /**
     * Generate a unique thread ID for IMAP emails
     */
    private function generateThreadId(array $emailData): string
    {
        // Use subject + from address + timestamp as thread identifier
        $identifier = md5(
            $emailData['subject'] . 
            $emailData['from_addr'] . 
            date('Y-m-d', strtotime($emailData['date']))
        );
        
        return "imap_thread_{$identifier}";
    }

    /**
     * Replace domain placeholders in provider settings
     */
    private function replaceDomainPlaceholders(string $template, string $email): string
    {
        $domain = substr(strrchr($email, "@"), 1);
        $parts = explode('.', $domain);
        $tld = array_pop($parts);
        $domainName = implode('.', $parts);

        return str_replace([
            'yourdomain.com',
            'your-domain.com',
            '{domain}',
            '{domain_name}.{tld}',
            '{domain}.{tld}'
        ], [
            $domain,
            $domain,
            $domainName,
            $domain,
            $domain
        ], $template);
    }

    /**
     * Update IMAP password for an existing account
     */
    public function updateImapPassword(Request $request, MailAccount $mailAccount)
    {
        $request->validate([
            'imap_password' => 'required|string|min:1'
        ]);

        try {
            // Update only the IMAP password
            $mailAccount->imap_password = $request->imap_password;
            $mailAccount->save();

            // Test the connection with new password
            $testResult = $this->imapService->testConnection($mailAccount);

            if ($testResult['success']) {
                return response()->json([
                    'success' => true,
                    'message' => '✅ IMAP password updated and connection verified!'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => '❌ Password updated but connection failed: ' . $testResult['message']
                ]);
            }

        } catch (\Exception $e) {
            Log::error('Failed to update IMAP password: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update IMAP password: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Add IMAP configuration to existing SMTP account
     */
    public function addImapToExisting(Request $request, MailAccount $mailAccount)
    {
        try {
            $this->authorize('admin', auth()->user());

            if ($mailAccount->connection_type !== 'smtp') {
                return redirect()->route('email.accounts')
                    ->with('error', 'Account is not an SMTP account.');
            }

            if ($mailAccount->hasImapConfig()) {
                return redirect()->route('email.accounts')
                    ->with('success', 'Account already has IMAP configuration.');
            }

            // Auto-resolve IMAP settings
            $imapHost = $this->imapService->resolveImapHost($mailAccount->email);
            
            $mailAccount->update([
                'imap_host' => $imapHost,
                'imap_port' => 993,
                'imap_encryption' => 'ssl',
                'imap_username' => $mailAccount->email,
                'imap_password' => $mailAccount->smtp_password, // Use same password
                'imap_folder' => 'INBOX',
            ]);

            Log::info('Added IMAP to existing SMTP account', [
                'email' => $mailAccount->email,
                'imap_host' => $imapHost
            ]);

            return redirect()->route('email.accounts')
                ->with('success', "IMAP configuration added to {$mailAccount->email}. You can now fetch emails!");

        } catch (\Exception $e) {
            Log::error('Failed to add IMAP to existing account: ' . $e->getMessage());
            return redirect()->route('email.accounts')
                ->with('error', 'Failed to add IMAP configuration: ' . $e->getMessage());
        }
    }

    /**
     * Show signatures management page
     */
    public function signatures(MailAccount $mailAccount)
    {
        if (!$mailAccount) {
            return redirect()->route('email.accounts')
                ->with('error', 'Mail account not found.');
        }
        
        $signatures = $mailAccount->emailSignatures()->orderBy('is_default', 'desc')->orderBy('created_at', 'desc')->get();
        return view('email.signatures', compact('mailAccount', 'signatures'));
    }

    /**
     * Get signatures for a mail account (AJAX)
     */
    public function getSignatures(MailAccount $mailAccount)
    {
        try {
            if (!$mailAccount) {
                return response()->json(['error' => 'Mail account not found'], 404);
            }
            
            $signatures = $mailAccount->emailSignatures()->active()->get();
            
            return response()->json([
                'success' => true,
                'signatures' => $signatures->map(function($signature) {
                    return [
                        'id' => $signature->id,
                        'signature_name' => $signature->signature_name,
                        'signature_html' => $signature->signature_html,
                        'signature_text' => $signature->signature_text,
                        'is_default' => $signature->is_default,
                    ];
                })
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to get signatures: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to load signatures'
            ], 500);
        }
    }

    /**
     * Store a new signature for a mail account
     */
    public function storeSignature(Request $request, MailAccount $mailAccount)
    {
        $request->validate([
            'signature_name' => 'required|string|max:255',
            'signature_html' => 'required|string',
            'signature_text' => 'required|string',
            'is_default' => 'boolean',
        ]);

        try {
            // If this is set as default, unset other defaults
            if ($request->boolean('is_default')) {
                $mailAccount->emailSignatures()->update(['is_default' => false]);
            }

            // Process signature HTML to ensure proper formatting
            $signatureHtml = $request->signature_html;
            
            // If the signature doesn't contain HTML tags, convert line breaks to <br> tags
            if (!preg_match('/<[^>]+>/', $signatureHtml)) {
                // Convert line breaks to <br> tags
                $signatureHtml = nl2br(htmlspecialchars($signatureHtml, ENT_QUOTES, 'UTF-8'));
            }

            $signature = $mailAccount->emailSignatures()->create([
                'signature_name' => $request->signature_name,
                'signature_html' => $signatureHtml,
                'signature_text' => $request->signature_text,
                'is_default' => $request->boolean('is_default'),
                'is_active' => true,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Signature created successfully',
                'signature' => [
                    'id' => $signature->id,
                    'signature_name' => $signature->signature_name,
                    'signature_html' => $signature->signature_html,
                    'signature_text' => $signature->signature_text,
                    'is_default' => $signature->is_default,
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to store signature: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to create signature: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update an existing signature
     */
    public function updateSignature(Request $request, MailAccount $mailAccount, EmailSignature $signature)
    {
        $request->validate([
            'signature_name' => 'required|string|max:255',
            'signature_html' => 'required|string',
            'signature_text' => 'required|string',
            'is_default' => 'boolean',
        ]);

        try {
            // If this is set as default, unset other defaults
            if ($request->boolean('is_default')) {
                $mailAccount->emailSignatures()->where('id', '!=', $signature->id)->update(['is_default' => false]);
            }

            // Process signature HTML to ensure proper formatting
            $signatureHtml = $request->signature_html;
            
            // If the signature doesn't contain HTML tags, convert line breaks to <br> tags
            if (!preg_match('/<[^>]+>/', $signatureHtml)) {
                // Convert line breaks to <br> tags
                $signatureHtml = nl2br(htmlspecialchars($signatureHtml, ENT_QUOTES, 'UTF-8'));
            }

            $signature->update([
                'signature_name' => $request->signature_name,
                'signature_html' => $signatureHtml,
                'signature_text' => $request->signature_text,
                'is_default' => $request->boolean('is_default'),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Signature updated successfully',
                'signature' => [
                    'id' => $signature->id,
                    'signature_name' => $signature->signature_name,
                    'signature_html' => $signature->signature_html,
                    'signature_text' => $signature->signature_text,
                    'is_default' => $signature->is_default,
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to update signature: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to update signature: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a signature
     */
    public function deleteSignature(MailAccount $mailAccount, EmailSignature $signature)
    {
        try {
            $signature->delete();

            return response()->json([
                'success' => true,
                'message' => 'Signature deleted successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to delete signature: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to delete signature: ' . $e->getMessage()
            ], 500);
        }
    }
}




