<?php

namespace App\Http\Controllers\Agent;

use App\Http\Controllers\Controller;
use App\Models\EmailDraft;
use App\Models\EmailSignature;
use App\Models\MailAccount;
use App\Models\Message;
use App\Models\Thread;
use App\Services\GmailService;
use App\Services\EnhancedReplyDraftService;
use App\Services\AIServiceManager;
use App\Services\SMTPEmailService;
use App\Services\IMAPEmailService;
use App\Services\UserContextService;
use App\Models\EmailProvider;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class EmailController extends Controller
{
    use AuthorizesRequests;

    private GmailService $gmailService;
    private EnhancedReplyDraftService $draftService;
    private SMTPEmailService $smtpService;
    private IMAPEmailService $imapService;
    private UserContextService $userContext;

    public function __construct(GmailService $gmailService, AIServiceManager $aiManager, SMTPEmailService $smtpService, IMAPEmailService $imapService, UserContextService $userContext)
    {
        $this->gmailService = $gmailService;
        $this->draftService = new EnhancedReplyDraftService($aiManager);
        $this->smtpService = $smtpService;
        $this->imapService = $imapService;
        $this->userContext = $userContext;
    }

    /**
     * Display email threads - Agent sees only their own threads
     */
    public function index(Request $request)
    {
        // Agent sees only their own threads
        $query = Thread::where('user_id', auth()->id());
        
        // Filter by mail account if specified
        if ($request->has('account') && $request->account !== 'all') {
            $query = $query->where('mail_account_id', $request->account);
        }
        
        // Optimize query by limiting messages loaded per thread
        // Only load the 5 most recent messages per thread to reduce query load
        $threads = $query
            ->with([
                'messages' => function ($query) {
                    $query->select('id', 'thread_id', 'direction', 'from_addr', 'snippet', 'date')
                          ->orderBy('date', 'desc')
                          ->limit(5);
                },
                'emailDrafts' => function ($query) {
                    $query->select('id', 'thread_id', 'draft_html', 'status', 'ai_meta', 'created_at')
                          ->orderBy('created_at', 'desc');
                },
                'mailAccount' => function ($query) {
                    $query->select('id', 'email');
                }
            ])
            ->select('id', 'user_id', 'mail_account_id', 'subject', 'participants', 'last_message_at', 'status')
            ->orderBy('last_message_at', 'desc')
            ->paginate(20);
        
        // Agent sees only their own mail accounts
        $mailAccounts = MailAccount::where('user_id', auth()->id())
            ->select('id', 'user_id', 'email', 'connection_type')
            ->get();
        
        return view('agent.email.index', compact('threads', 'mailAccounts'));
    }

    /**
     * Show individual email thread with full content
     */
    public function show(Thread $thread)
    {
        // Verify thread belongs to agent
        if ($thread->user_id !== auth()->id()) {
            abort(403, 'You do not have permission to view this thread.');
        }

        // Load messages with thread
        $thread->load('messages');
        
        return view('agent.email.show', compact('thread'));
    }

    /**
     * Bulk delete selected threads - Agent can delete only their own threads
     */
    public function bulkDelete(Request $request)
    {
        try {
            $threadIds = $request->input('thread_ids', []);
            
            if (empty($threadIds)) {
                return redirect()->route('agent.email.index')
                    ->with('error', 'No threads selected for deletion.');
            }

            // Agent can only delete their own threads
            $threadsToDelete = Thread::where('user_id', auth()->id())
                ->whereIn('id', $threadIds)
                ->get();

            if ($threadsToDelete->count() !== count($threadIds)) {
                return redirect()->route('agent.email.index')
                    ->with('error', 'Some selected threads could not be found or you do not have permission to delete them.');
            }

            $deletedCount = 0;
            foreach ($threadsToDelete as $thread) {
                // Delete associated messages first
                $thread->messages()->delete();
                
                // Delete associated drafts
                $thread->emailDrafts()->delete();
                
                // Delete the thread
                $thread->delete();
                $deletedCount++;
            }

            return redirect()->route('agent.email.index')
                ->with('success', "Successfully deleted {$deletedCount} email thread(s).");

        } catch (\Exception $e) {
            Log::error('Failed to bulk delete threads: ' . $e->getMessage());
            return redirect()->route('agent.email.index')
                ->with('error', 'Failed to delete threads: ' . $e->getMessage());
        }
    }

    /**
     * Delete individual thread - Agent can delete only their own threads
     */
    public function destroyThread(Thread $thread)
    {
        try {
            // Verify thread belongs to agent
            if ($thread->user_id !== auth()->id()) {
                abort(403, 'You do not have permission to delete this thread.');
            }

            // Delete associated messages first
            $thread->messages()->delete();
            
            // Delete associated drafts
            $thread->emailDrafts()->delete();
            
            // Delete the thread
            $thread->delete();

            return redirect()->route('agent.email.index')
                ->with('success', 'Email thread deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to delete thread: ' . $e->getMessage());
            return redirect()->route('agent.email.index')
                ->with('error', 'Failed to delete thread: ' . $e->getMessage());
        }
    }
}
