<?php

namespace App\Http\Controllers;

use App\Services\AIServiceManager;
use App\Services\AIProviders\OpenAIProvider;
use App\Services\AIProviders\GeminiProvider;
use App\Services\AIProviders\AnthropicProvider;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class AIController extends Controller
{
    private AIServiceManager $aiManager;

    public function __construct(AIServiceManager $aiManager)
    {
        $this->aiManager = $aiManager;
    }

    /**
     * Show AI provider settings page
     */
    public function settings()
    {
        $providerInfo = $this->aiManager->getProviderInfo();
        
        return view('ai.settings', [
            'providers' => $providerInfo,
            'defaultProvider' => config('services.ai.default_provider', 'gemini'),
        ]);
    }

    /**
     * Get available AI providers
     */
    public function getProviders(): JsonResponse
    {
        return response()->json([
            'success' => true,
            'providers' => $this->aiManager->getProviderInfo(),
            'default' => config('services.ai.default_provider', 'gemini'),
        ]);
    }

    /**
     * Test AI provider connection
     */
    public function testProvider(Request $request): JsonResponse
    {
        $request->validate([
            'provider' => 'required|string|in:openai,gemini,anthropic',
        ]);

        try {
            $provider = $this->createProviderInstance($request->provider);
            $result = $provider->testConnection();

            return response()->json([
                'success' => $result['success'],
                'message' => $result['success'] ? 'Connection successful' : 'Connection failed',
                'details' => $result,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Provider not available: ' . $e->getMessage(),
            ], 400);
        }
    }

    /**
     * Test all AI providers
     */
    public function testAllProviders(): JsonResponse
    {
        $results = $this->aiManager->testAllProviders();

        return response()->json([
            'success' => true,
            'results' => $results,
        ]);
    }

    /**
     * Update AI provider settings
     */
    public function updateSettings(Request $request): JsonResponse
    {
        $request->validate([
            'default_provider' => 'required|string|in:openai,gemini,anthropic',
            'providers' => 'required|array',
            'providers.*.enabled' => 'required|boolean',
        ]);

        $user = auth()->user();
        $tenant = $user?->tenant;

        if (!$tenant && $user?->role === 'super_admin') {
            $tenant = Tenant::where('slug', 'default')->first();
        }

        if (!$user || !$tenant) {
            return response()->json([
                'success' => false,
                'message' => 'Unable to update settings for this account.',
            ], 403);
        }

        $normalizedProviders = [];
        foreach ($request->providers as $key => $provider) {
            $normalizedProviders[$key] = [
                'enabled' => $key === $request->default_provider,
            ];
        }

        $settings = $tenant->settings ?? [];
        $settings['ai_settings'] = [
            'default_provider' => $request->default_provider,
            'providers' => $normalizedProviders,
        ];

        $tenant->update(['settings' => $settings]);

        return response()->json([
            'success' => true,
            'message' => 'Settings updated successfully.',
            'settings' => [
                'default_provider' => $request->default_provider,
                'providers' => $normalizedProviders,
            ],
        ]);
    }

    /**
     * Create a provider instance without checking tenant enablement
     */
    private function createProviderInstance(string $providerKey)
    {
        return match ($providerKey) {
            'openai' => new OpenAIProvider(),
            'gemini' => new GeminiProvider(),
            'anthropic' => new AnthropicProvider(),
            default => throw new \InvalidArgumentException("Unknown AI provider: {$providerKey}")
        };
    }
}
