<?php

namespace App\Console\Commands;

use App\Models\User;
use App\Models\Tenant;
use App\Models\MailAccount;
use App\Models\Thread;
use App\Models\EmailDraft;
use App\Services\UserContextService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Auth;

class TestMultiTenantIsolation extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'test:multi-tenant-isolation';

    /**
     * The console command description.
     */
    protected $description = 'Test multi-tenant data isolation';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🧪 Testing Multi-Tenant Data Isolation...');
        $this->newLine();

        // Get all users
        $users = User::with('tenant')->get();
        
        if ($users->count() < 2) {
            $this->error('Need at least 2 users to test isolation. Current users: ' . $users->count());
            return Command::FAILURE;
        }

        $this->info('Found ' . $users->count() . ' users:');
        foreach ($users as $user) {
            $this->info("  - {$user->name} ({$user->email}) - Role: {$user->role} - Tenant: " . ($user->tenant ? $user->tenant->name : 'None'));
        }
        $this->newLine();

        // Test data isolation for each user
        foreach ($users as $user) {
            $this->info("🔍 Testing data isolation for: {$user->name}");
            
            // Simulate user login
            Auth::login($user);
            
            $userContext = new UserContextService();
            
            // Test mail accounts
            $mailAccounts = $userContext->getUserMailAccounts()->get();
            $this->info("  📧 Mail Accounts: {$mailAccounts->count()}");
            
            // Test threads
            $threads = $userContext->getUserThreads()->get();
            $this->info("  🧵 Threads: {$threads->count()}");
            
            // Test email drafts
            $drafts = $userContext->getUserEmailDrafts()->get();
            $this->info("  📝 Email Drafts: {$drafts->count()}");
            
            // Test locations
            $locations = $userContext->getUserLocations()->get();
            $this->info("  📍 Locations: {$locations->count()}");
            
            // Verify data belongs to user
            $allDataBelongsToUser = true;
            
            foreach ($mailAccounts as $account) {
                if ($account->user_id !== $user->id) {
                    $allDataBelongsToUser = false;
                    $this->error("    ❌ Mail account {$account->email} belongs to user {$account->user_id}, not {$user->id}");
                }
            }
            
            foreach ($threads as $thread) {
                if ($thread->user_id !== $user->id) {
                    $allDataBelongsToUser = false;
                    $this->error("    ❌ Thread {$thread->subject} belongs to user {$thread->user_id}, not {$user->id}");
                }
            }
            
            foreach ($drafts as $draft) {
                if ($draft->user_id !== $user->id) {
                    $allDataBelongsToUser = false;
                    $this->error("    ❌ Email draft belongs to user {$draft->user_id}, not {$user->id}");
                }
            }
            
            foreach ($locations as $location) {
                if ($location->user_id !== $user->id) {
                    $allDataBelongsToUser = false;
                    $this->error("    ❌ Location {$location->display_name} belongs to user {$location->user_id}, not {$user->id}");
                }
            }
            
            if ($allDataBelongsToUser) {
                $this->info("  ✅ All data properly isolated for {$user->name}");
            } else {
                $this->error("  ❌ Data isolation failed for {$user->name}");
            }
            
            $this->newLine();
        }

        // Test super admin access
        $superAdmin = $users->where('role', 'super_admin')->first();
        if ($superAdmin) {
            $this->info("🔍 Testing Super Admin access for: {$superAdmin->name}");
            
            Auth::login($superAdmin);
            $userContext = new UserContextService();
            
            // Super admin should see all data
            $allMailAccounts = MailAccount::all();
            $allThreads = Thread::all();
            $allDrafts = EmailDraft::all();
            $allLocations = \App\Models\Location::all();
            
            $this->info("  📧 All Mail Accounts: {$allMailAccounts->count()}");
            $this->info("  🧵 All Threads: {$allThreads->count()}");
            $this->info("  📝 All Email Drafts: {$allDrafts->count()}");
            $this->info("  📍 All Locations: {$allLocations->count()}");
            
            $this->info("  ✅ Super admin can access all data");
        }

        $this->newLine();
        $this->info('🎉 Multi-tenant isolation test completed!');
        
        return Command::SUCCESS;
    }
}
