<?php

namespace App\Console\Commands;

use App\Services\SecurityService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class SecurityAudit extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'security:audit {--fix : Automatically fix security issues} {--type=all : Type of audit (sql|xss|all)}';

    /**
     * The console command description.
     */
    protected $description = 'Comprehensive security audit for SQL injection and XSS vulnerabilities';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔒 Starting Comprehensive Security Audit...');
        
        $type = $this->option('type');
        $fix = $this->option('fix');
        
        $issues = [];
        
        if ($type === 'all' || $type === 'sql') {
            $issues = array_merge($issues, $this->auditSqlInjection());
        }
        
        if ($type === 'all' || $type === 'xss') {
            $issues = array_merge($issues, $this->auditXssVulnerabilities());
        }
        
        $this->displayResults($issues, $fix);
        
        return Command::SUCCESS;
    }

    /**
     * Audit for SQL injection vulnerabilities
     */
    private function auditSqlInjection(): array
    {
        $this->info('🔍 Auditing SQL injection vulnerabilities...');
        
        $issues = [];
        $files = $this->getPhpFiles();
        
        foreach ($files as $file) {
            $content = File::get($file);
            
            // Check for raw SQL queries
            if (preg_match_all('/DB::(raw|select|statement|unprepared)\s*\(\s*["\']([^"\']+)["\']/', $content, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    $issues[] = [
                        'type' => 'SQL_INJECTION',
                        'severity' => 'HIGH',
                        'file' => $file,
                        'line' => $this->getLineNumber($content, $match[0]),
                        'description' => 'Raw SQL query detected: ' . substr($match[2], 0, 50) . '...',
                        'code' => $match[0]
                    ];
                }
            }
            
            // Check for whereRaw usage
            if (preg_match_all('/whereRaw\s*\(\s*["\']([^"\']+)["\']/', $content, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    $issues[] = [
                        'type' => 'SQL_INJECTION',
                        'severity' => 'MEDIUM',
                        'file' => $file,
                        'line' => $this->getLineNumber($content, $match[0]),
                        'description' => 'whereRaw usage detected: ' . substr($match[1], 0, 50) . '...',
                        'code' => $match[0]
                    ];
                }
            }
            
            // Check for direct variable interpolation in SQL
            if (preg_match_all('/["\']\s*\.\s*\$[a-zA-Z_][a-zA-Z0-9_]*\s*\.\s*["\']/', $content, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    $issues[] = [
                        'type' => 'SQL_INJECTION',
                        'severity' => 'HIGH',
                        'file' => $file,
                        'line' => $this->getLineNumber($content, $match[0]),
                        'description' => 'Variable interpolation in SQL string detected',
                        'code' => $match[0]
                    ];
                }
            }
        }
        
        return $issues;
    }

    /**
     * Audit for XSS vulnerabilities
     */
    private function auditXssVulnerabilities(): array
    {
        $this->info('🔍 Auditing XSS vulnerabilities...');
        
        $issues = [];
        $viewFiles = $this->getViewFiles();
        
        foreach ($viewFiles as $file) {
            $content = File::get($file);
            
            // Check for unescaped output
            if (preg_match_all('/\{!!\s*([^}]+)\s*!!\}/', $content, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    $output = trim($match[1]);
                    
                    // Skip if it's using strip_tags or htmlspecialchars
                    if (strpos($output, 'strip_tags') !== false || 
                        strpos($output, 'htmlspecialchars') !== false ||
                        strpos($output, 'e(') !== false) {
                        continue;
                    }
                    
                    $issues[] = [
                        'type' => 'XSS',
                        'severity' => 'HIGH',
                        'file' => $file,
                        'line' => $this->getLineNumber($content, $match[0]),
                        'description' => 'Unescaped output detected: ' . substr($output, 0, 50) . '...',
                        'code' => $match[0]
                    ];
                }
            }
            
            // Check for echo without escaping
            if (preg_match_all('/echo\s+[^;]+;/', $content, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    if (strpos($match[0], 'htmlspecialchars') === false && 
                        strpos($match[0], 'e(') === false) {
                        $issues[] = [
                            'type' => 'XSS',
                            'severity' => 'MEDIUM',
                            'file' => $file,
                            'line' => $this->getLineNumber($content, $match[0]),
                            'description' => 'Echo without escaping detected',
                            'code' => $match[0]
                        ];
                    }
                }
            }
            
            // Check for print without escaping
            if (preg_match_all('/print\s+[^;]+;/', $content, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    if (strpos($match[0], 'htmlspecialchars') === false && 
                        strpos($match[0], 'e(') === false) {
                        $issues[] = [
                            'type' => 'XSS',
                            'severity' => 'MEDIUM',
                            'file' => $file,
                            'line' => $this->getLineNumber($content, $match[0]),
                            'description' => 'Print without escaping detected',
                            'code' => $match[0]
                        ];
                    }
                }
            }
        }
        
        return $issues;
    }

    /**
     * Get all PHP files
     */
    private function getPhpFiles(): array
    {
        $files = [];
        $directories = [
            app_path(),
            base_path('routes'),
            base_path('database/migrations'),
        ];
        
        foreach ($directories as $directory) {
            if (is_dir($directory)) {
                $files = array_merge($files, File::allFiles($directory));
            }
        }
        
        return array_filter($files, function($file) {
            return $file->getExtension() === 'php';
        });
    }

    /**
     * Get all view files
     */
    private function getViewFiles(): array
    {
        $viewPath = resource_path('views');
        if (!is_dir($viewPath)) {
            return [];
        }
        
        return File::allFiles($viewPath);
    }

    /**
     * Get line number of code in content
     */
    private function getLineNumber(string $content, string $code): int
    {
        $lines = explode("\n", $content);
        foreach ($lines as $index => $line) {
            if (strpos($line, $code) !== false) {
                return $index + 1;
            }
        }
        return 0;
    }

    /**
     * Display audit results
     */
    private function displayResults(array $issues, bool $fix): void
    {
        $this->newLine();
        
        if (empty($issues)) {
            $this->info('✅ No security vulnerabilities found!');
            return;
        }

        // Group by severity
        $highIssues = array_filter($issues, fn($issue) => $issue['severity'] === 'HIGH');
        $mediumIssues = array_filter($issues, fn($issue) => $issue['severity'] === 'MEDIUM');
        $lowIssues = array_filter($issues, fn($issue) => $issue['severity'] === 'LOW');

        $this->warn("🚨 Found " . count($issues) . " security issues:");
        $this->newLine();

        // Display high severity issues
        if (!empty($highIssues)) {
            $this->error("🔴 HIGH SEVERITY (" . count($highIssues) . "):");
            foreach ($highIssues as $issue) {
                $this->line("  📄 " . str_replace(base_path() . '/', '', $issue['file']));
                $this->line("  📍 Line " . $issue['line'] . ": " . $issue['description']);
                $this->line("  🔧 " . $issue['code']);
                $this->newLine();
            }
        }

        // Display medium severity issues
        if (!empty($mediumIssues)) {
            $this->warn("🟡 MEDIUM SEVERITY (" . count($mediumIssues) . "):");
            foreach ($mediumIssues as $issue) {
                $this->line("  📄 " . str_replace(base_path() . '/', '', $issue['file']));
                $this->line("  📍 Line " . $issue['line'] . ": " . $issue['description']);
                $this->line("  🔧 " . $issue['code']);
                $this->newLine();
            }
        }

        // Display low severity issues
        if (!empty($lowIssues)) {
            $this->info("🟢 LOW SEVERITY (" . count($lowIssues) . "):");
            foreach ($lowIssues as $issue) {
                $this->line("  📄 " . str_replace(base_path() . '/', '', $issue['file']));
                $this->line("  📍 Line " . $issue['line'] . ": " . $issue['description']);
                $this->newLine();
            }
        }

        if ($fix) {
            $this->info("🔧 Auto-fix functionality not yet implemented");
        } else {
            $this->warn("💡 Run with --fix to attempt automatic fixes");
        }

        $this->newLine();
        $this->info("🛡️ Security Recommendations:");
        $this->line("   • Use parameterized queries for database operations");
        $this->line("   • Always escape output with {{ }} or e() helper");
        $this->line("   • Use {!! !!} only for trusted HTML content");
        $this->line("   • Validate and sanitize all user input");
        $this->line("   • Use Laravel's built-in security features");
    }
}
