<?php

namespace App\Console\Commands;

use App\Jobs\ProcessEmailIngestion;
use App\Models\MailAccount;
use Illuminate\Console\Command;

class QueueEmailIngestion extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'email:queue-ingestion {--account= : Specific account ID to process} {--type=imap : Ingestion type (imap|oauth)}';

    /**
     * The console command description.
     */
    protected $description = 'Queue email ingestion jobs for background processing';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $accountId = $this->option('account');
        $ingestionType = $this->option('type');

        if ($accountId) {
            $account = MailAccount::find($accountId);
            if (!$account) {
                $this->error("Account with ID {$accountId} not found.");
                return Command::FAILURE;
            }
            $accounts = collect([$account]);
        } else {
            $accounts = MailAccount::where('is_active', true)->get();
        }

        $queuedCount = 0;

        foreach ($accounts as $account) {
            if ($ingestionType === 'imap' && !$account->hasImapConfig()) {
                $this->warn("Skipping {$account->email} - No IMAP configuration");
                continue;
            }

            if ($ingestionType === 'oauth' && !$account->oauth_connection_id) {
                $this->warn("Skipping {$account->email} - No OAuth connection");
                continue;
            }

            ProcessEmailIngestion::dispatch($account, $ingestionType);
            $queuedCount++;

            $this->info("Queued {$ingestionType} ingestion for {$account->email}");
        }

        $this->info("Queued {$queuedCount} email ingestion jobs");
        return Command::SUCCESS;
    }
}
