<?php

namespace App\Console\Commands;

use App\Models\MailAccount;
use App\Jobs\IngestEmailsJob;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

/**
 * Ingest All Emails Command
 *
 * Dispatches email ingestion jobs for all mail accounts in the system.
 * Scheduled to run every 15 minutes via Laravel Task Scheduler.
 *
 * Reference: AUTO_INGESTION.md Section 3.3
 */
class IngestAllEmailsCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'emails:ingest-all';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Dispatch email ingestion jobs for all mail accounts';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Starting email ingestion for all accounts...');

        // Get all mail accounts (bypassing global scopes to get ALL accounts system-wide)
        // Each job is scoped to a specific account, ensuring multi-tenant safety
        $mailAccounts = MailAccount::withoutGlobalScopes()->with('user.tenant')->get();

        if ($mailAccounts->count() === 0) {
            $this->warn('No mail accounts found in the system.');
            Log::warning('IngestAllEmailsCommand: No mail accounts found');
            return Command::SUCCESS;
        }

        $this->info("Found {$mailAccounts->count()} mail account(s). Checking tenant intervals...");

        $jobsDispatched = 0;
        $jobsSkipped = 0;

        // Group accounts by tenant to check intervals
        $accountsByTenant = $mailAccounts->groupBy('user.tenant_id');

        foreach ($accountsByTenant as $tenantId => $tenantAccounts) {
            $tenant = $tenantAccounts->first()->user->tenant;

            // Get tenant's configured interval (default: 15 minutes)
            $interval = $tenant->settings['email_ingestion_interval'] ?? 15;

            // Check if enough time has passed since last ingestion
            $cacheKey = "email_ingestion_last_run_tenant_{$tenantId}";
            $lastRun = \Illuminate\Support\Facades\Cache::get($cacheKey);

            if ($lastRun && now()->diffInMinutes($lastRun) < $interval) {
                $this->line("⏭ Skipping tenant {$tenant->name}: interval not reached ({$interval} min)");
                $jobsSkipped += $tenantAccounts->count();
                continue;
            }

            // Dispatch jobs for this tenant's accounts
            foreach ($tenantAccounts as $mailAccount) {
                try {
                    IngestEmailsJob::dispatch($mailAccount);
                    $jobsDispatched++;
                    $this->line("✓ Dispatched job for: {$mailAccount->email}");

                } catch (\Exception $e) {
                    $this->error("✗ Failed to dispatch job for {$mailAccount->email}: {$e->getMessage()}");
                    Log::error('IngestAllEmailsCommand: Failed to dispatch job', [
                        'account_email' => $mailAccount->email,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            // Update last run timestamp for this tenant
            \Illuminate\Support\Facades\Cache::put($cacheKey, now(), now()->addDay());
        }

        $this->info("Dispatched {$jobsDispatched} job(s), skipped {$jobsSkipped} (interval not reached).");

        Log::info('IngestAllEmailsCommand: Completed', [
            'total_accounts' => $mailAccounts->count(),
            'jobs_dispatched' => $jobsDispatched,
            'jobs_skipped' => $jobsSkipped
        ]);

        return Command::SUCCESS;
    }
}
