<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class CsrfAudit extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'security:csrf-audit {--fix : Automatically fix missing CSRF tokens}';

    /**
     * The console command description.
     */
    protected $description = 'Audit all forms for CSRF protection and optionally fix missing tokens';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Starting CSRF Security Audit...');
        
        $viewFiles = $this->getViewFiles();
        $issues = [];
        $fixed = 0;

        foreach ($viewFiles as $file) {
            $content = File::get($file);
            $forms = $this->findForms($content);
            
            foreach ($forms as $form) {
                $hasCsrf = $this->hasCsrfToken($form);
                $isAjax = $this->isAjaxForm($form);
                
                if (!$hasCsrf && !$isAjax) {
                    $issues[] = [
                        'file' => $file,
                        'form' => $form,
                        'line' => $this->getLineNumber($content, $form)
                    ];
                    
                    if ($this->option('fix')) {
                        $this->fixCsrfToken($file, $form);
                        $fixed++;
                    }
                }
            }
        }

        $this->displayResults($issues, $fixed);
        
        return Command::SUCCESS;
    }

    /**
     * Get all Blade view files
     */
    private function getViewFiles(): array
    {
        $viewPath = resource_path('views');
        return File::allFiles($viewPath);
    }

    /**
     * Find all form elements in content
     */
    private function findForms(string $content): array
    {
        preg_match_all('/<form[^>]*>.*?<\/form>/s', $content, $matches);
        return $matches[0] ?? [];
    }

    /**
     * Check if form has CSRF token
     */
    private function hasCsrfToken(string $form): bool
    {
        return Str::contains($form, '@csrf') || 
               Str::contains($form, 'csrf_token()') ||
               Str::contains($form, '_token') ||
               Str::contains($form, 'X-CSRF-TOKEN');
    }

    /**
     * Check if form is AJAX-only (doesn't need @csrf)
     */
    private function isAjaxForm(string $form): bool
    {
        return Str::contains($form, 'onsubmit=') ||
               Str::contains($form, 'fetch(') ||
               Str::contains($form, 'axios.') ||
               Str::contains($form, 'XMLHttpRequest');
    }

    /**
     * Get line number of form in content
     */
    private function getLineNumber(string $content, string $form): int
    {
        $lines = explode("\n", $content);
        foreach ($lines as $index => $line) {
            if (Str::contains($line, '<form')) {
                return $index + 1;
            }
        }
        return 0;
    }

    /**
     * Fix missing CSRF token
     */
    private function fixCsrfToken(string $file, string $form): void
    {
        $content = File::get($file);
        $fixedForm = $this->addCsrfToken($form);
        $newContent = str_replace($form, $fixedForm, $content);
        File::put($file, $newContent);
    }

    /**
     * Add CSRF token to form
     */
    private function addCsrfToken(string $form): string
    {
        // Find the opening form tag
        if (preg_match('/<form([^>]*)>/', $form, $matches)) {
            $formTag = $matches[0];
            $formAttributes = $matches[1];
            
            // Add @csrf after the opening form tag
            $newFormTag = $formTag . "\n            @csrf";
            return str_replace($formTag, $newFormTag, $form);
        }
        
        return $form;
    }

    /**
     * Display audit results
     */
    private function displayResults(array $issues, int $fixed): void
    {
        $this->newLine();
        
        if (empty($issues)) {
            $this->info('✅ All forms have CSRF protection!');
            return;
        }

        $this->warn("⚠️  Found " . count($issues) . " forms without CSRF protection:");
        $this->newLine();

        foreach ($issues as $issue) {
            $this->line("📄 File: " . str_replace(resource_path('views/'), '', $issue['file']));
            $this->line("📍 Line: " . $issue['line']);
            $this->line("🔧 Form: " . Str::limit(strip_tags($issue['form']), 100));
            $this->newLine();
        }

        if ($fixed > 0) {
            $this->info("✅ Fixed {$fixed} forms with missing CSRF tokens");
        } else {
            $this->warn("💡 Run with --fix to automatically add missing CSRF tokens");
        }

        $this->newLine();
        $this->info("🔒 CSRF Security Recommendations:");
        $this->line("   • All forms should include @csrf directive");
        $this->line("   • AJAX requests should include X-CSRF-TOKEN header");
        $this->line("   • Use csrf_token() helper for custom implementations");
        $this->line("   • Test forms to ensure CSRF protection works");
    }
}
