# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

ReplyPilot is a **multi-tenant** Laravel 12 application for AI-powered reply management for Google Business Profile reviews and Gmail emails with human-in-the-loop approval. It supports multiple AI providers (ChatGPT, Anthropic, Gemini) with Gemini as the default.

**Core Principle**: Never auto-post/send - always requires human approval before posting reviews or sending emails.

## Common Commands

### Development
```bash
# Start full development environment (server, queue, logs, vite)
composer dev

# Start queue worker only
php artisan queue:work

# Start development server only
php artisan serve

# Build frontend assets
npm run build

# Development mode for frontend
npm run dev
```

### Testing
```bash
# Run all tests
composer test
# or
php artisan test

# Test multi-tenant isolation
php artisan test:multi-tenant-isolation

# Code formatting with Laravel Pint
./vendor/bin/pint
```

### Installation & Setup
```bash
# Initial application setup (runs migrations, creates admin user)
php artisan app:install

# Check admin users
php artisan check:admin-users

# Reset admin password
php artisan reset:admin-password
```

### AI Provider Management
```bash
# Test all configured AI providers
php artisan ai:test-providers

# Test specific OpenAI configuration
php artisan openai:test
```

### Data Ingestion
```bash
# Queue email ingestion (runs in background via jobs)
php artisan email:queue-ingestion

# Auto-ingest emails (scheduled task)
php artisan email:auto-ingest

# Auto-ingest reviews (scheduled task)
php artisan reviews:auto-ingest

# Clean up old emails
php artisan email:cleanup-old
```

### Scheduled Tasks
```bash
# Run Laravel scheduler continuously (recommended for development)
php artisan schedule:work

# Run scheduler once (useful for cron jobs)
php artisan schedule:run

# Check status of scheduled tasks
php artisan schedule:status

# List all scheduled tasks
php artisan schedule:list
```

**Active Scheduled Tasks** (configured in routes/console.php):
- Email auto-ingestion: Every minute (with internal 15-minute tenant interval)
- Review auto-ingestion: Every minute (with internal 60-minute tenant interval)
- Email cleanup: Daily at 2:00 AM

### Security & Maintenance
```bash
# Run security audit
php artisan security:audit

# Run CSRF audit
php artisan csrf:audit

# Optimize for production
php artisan production:optimize
```

## Architecture Overview

### Multi-Tenancy & Role-Based Access Control

The application supports a **three-tier role hierarchy**:
- **Super Admin**: Full system access across all tenants
- **Tenant Admin**: Full access to their tenant's data, can manage tenant users
- **Agent**: Limited access - can draft but cannot approve/post

**Key Implementation Details**:
- The `BelongsToUser` trait (app/Traits/BelongsToUser.php) provides global scopes for data isolation
- All models use `user_id` and `tenant_id` for multi-tenant filtering
- Route prefixes: `/super-admin/*`, `/tenant-admin/*`, `/agent/*` with corresponding middleware
- Middleware: `SuperAdminMiddleware`, `TenantAdminMiddleware`, `AgentMiddleware`

### AI Provider Architecture

The application uses a **provider pattern** for AI services:
- **AIServiceManager** (app/Services/AIServiceManager.php) manages multiple AI providers
- **AIProviderInterface** contract defines the provider API
- Three implementations: `OpenAIProvider`, `GeminiProvider`, `AnthropicProvider`
- Configuration in config/services.php controls which providers are enabled
- Default provider set via `AI_DEFAULT_PROVIDER` env variable (defaults to 'gemini')

### Data Flow for Review & Email Processing

1. **Ingestion**: Data fetched from Google APIs (Gmail/Business Profile) → stored in database
2. **Draft Generation**: AI provider generates draft reply → stored as `ReviewDraft` or `EmailDraft`
3. **Approval**: Admin/Tenant Admin approves draft (escalation keywords require admin)
4. **Posting**: Approved draft posted back to Google API

**Important**: Processing happens via **queued jobs** (`ProcessEmailIngestion`, `ProcessReviewIngestion`) to handle API rate limits and long-running operations.

### Escalation System

Located in `ReplyDraftService::checkEscalationKeywords()`:
- Keywords: 'injury', 'allergic', 'lawsuit', 'chargeback', 'refund'
- When detected, draft is marked for escalation (admin-only approval)
- Prevents agents from handling sensitive content

### Google OAuth & API Integration

- **OAuth Flow**: Handled by `GoogleOAuthController` and Gmail module
- **Token Storage**: Encrypted tokens in `oauth_connections` table
- **Services**:
  - `GoogleBusinessProfile` service for reviews
  - `GmailService` for email operations
  - `GoogleClientFactory` for creating authenticated Google API clients
- **Required Scopes**: gmail.readonly, gmail.send, business.manage

### Email Account Support

The application supports **two types of email accounts**:

1. **Gmail OAuth Accounts**:
   - Uses Google OAuth for authentication
   - Requires `oauth_connection_id` in `mail_accounts` table
   - Provides access to Gmail API for sending/receiving
   - Tokens auto-refresh

2. **SMTP/IMAP Accounts**:
   - Direct SMTP connection for sending (via `SMTPEmailService`)
   - IMAP connection for receiving (via `IMAPEmailService`)
   - Stores encrypted credentials in `mail_accounts` table
   - Can add IMAP to existing SMTP accounts
   - Services: `app/Modules/SMTP/Services/SMTPService.php`

**Key Implementation**: `mail_accounts.oauth_connection_id` is nullable - NULL means SMTP/IMAP account, non-NULL means Gmail OAuth account.

### Modular Architecture

The application uses Laravel service providers for modular features:
- **Reviews Module**: `app/Modules/Reviews/` - Google Business Profile integration
- **Gmail Module**: `app/Modules/Gmail/` - Gmail integration
- **SMTP Module**: `app/Modules/SMTP/` - SMTP/IMAP email accounts
- Each module has its own routes, controllers, services, and views

### Database Schema Key Points

- **UUIDs**: All primary keys use UUIDs (not auto-increment integers)
- **Encrypted Fields**: OAuth tokens stored encrypted at database level
- **UTF8MB4**: All tables use utf8mb4 collation for emoji support
- **Key Tables**:
  - `users` - Authentication with role column
  - `tenants` - Multi-tenant isolation
  - `oauth_connections` - Google OAuth tokens
  - `reviews` + `review_drafts` - GBP review workflow
  - `threads` + `messages` + `email_drafts` - Email workflow
  - `mail_accounts` - Email accounts (Gmail OAuth or SMTP/IMAP)

## Important Guardrails

1. **Word Limits**: Reviews ≤ 80 words, Emails ≤ 120 words (enforced in `ReplyDraftService`)
2. **Human-in-the-Loop**: Never bypass approval step - this is a core feature
3. **Data Isolation**: Always use scopes from `BelongsToUser` trait for tenant isolation
4. **No Message Body Logging**: Never log email/review content bodies for privacy
5. **Automatic Token Refresh**: OAuth tokens auto-refresh; handle gracefully
6. **Email Retention**: Old emails are automatically cleaned up (default 24 hours, configurable via `EMAIL_RETENTION_HOURS`)

## Environment Configuration

Key environment variables (see .env.example for full list):

**AI Provider Configuration**:
- `AI_DEFAULT_PROVIDER` - Which AI service to use (openai/gemini/anthropic)
- `AI_GEMINI_ENABLED` - Enable Gemini (default: true)
- `AI_OPENAI_ENABLED` - Enable OpenAI (default: false)
- `AI_ANTHROPIC_ENABLED` - Enable Anthropic (default: false)
- `GEMINI_API_KEY` - Google Gemini API key
- `GEMINI_MODEL` - Model to use (default: gemini-2.5-pro)
- `OPENAI_API_KEY` - OpenAI API key
- `OPENAI_MODEL` - Model to use (default: gpt-4o-mini)
- `ANTHROPIC_API_KEY` - Anthropic API key
- `ANTHROPIC_MODEL` - Model to use (default: claude-3-haiku-20240307)

**Google OAuth Configuration**:
- `GOOGLE_CLIENT_ID` - OAuth client ID from Google Cloud Console
- `GOOGLE_CLIENT_SECRET` - OAuth client secret
- `GOOGLE_REDIRECT_URI` - OAuth callback URL (e.g., ${APP_URL}/oauth/google/callback)
- `GOOGLE_ALLOWED_SCOPES` - Required scopes (gmail.readonly, gmail.send, business.manage)

**Email Retention**:
- `EMAIL_RETENTION_HOURS` - Hours to keep ingested emails (default: 24)

**System Configuration**:
- `QUEUE_CONNECTION=database` - Queue driver (required for background jobs)
- `SESSION_DRIVER=database` - Session storage
- `DB_CONNECTION=mysql` - Default database (can use MySQL/PostgreSQL)

## Production Deployment

### Required Services
For production, ensure these services are running continuously:
```bash
# Queue worker (processes background jobs)
php artisan queue:work --daemon

# Scheduler (runs scheduled tasks)
php artisan schedule:work
# OR set up system cron:
* * * * * cd /path/to/replypilot && php artisan schedule:run >> /dev/null 2>&1
```

### Production Optimization
```bash
# Run all production optimizations at once
php artisan production:optimize

# This includes:
# - Config caching
# - Route caching
# - View caching
# - Composer optimization
# - Database optimization
```

### Supervisor Configuration (Recommended)
For production environments, use Supervisor to keep queue workers running:
```ini
[program:replypilot-queue]
process_name=%(program_name)s_%(process_num)02d
command=php /path/to/replypilot/artisan queue:work --sleep=3 --tries=3 --daemon
autostart=true
autorestart=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/path/to/replypilot/storage/logs/worker.log
```

## Security Features

### SSL Verification
SSL verification is controlled by the `SecureHttpClient` helper class:
- **Production/Staging**: SSL always enabled
- **Local Development**: Disabled only when `APP_ENV=local` AND `APP_DEBUG=true`

```php
use App\Support\SecureHttpClient;

// Create secure HTTP client
$client = SecureHttpClient::create();

// Get options for Laravel HTTP facade
Http::withOptions(SecureHttpClient::getHttpOptions())->get(...);
```

### Rate Limiting
API rate limiting middleware available for AI, Google, and email endpoints:

```php
// In routes
Route::middleware(['auth', 'rate.api:ai'])->group(function () {
    // AI endpoints - 60 requests/minute
});

Route::middleware(['auth', 'rate.api:google'])->group(function () {
    // Google API endpoints - 100 requests/minute
});
```

### Multi-Tenant Isolation
The `BelongsToUser` trait provides automatic data isolation:

- **Super Admin**: Access to all data across all tenants
- **Tenant Admin**: Access to all data within their tenant
- **Agent**: Access only to their own data

Models using this trait automatically:
- Filter queries by tenant_id and user_id based on role
- Set user_id and tenant_id on new records

### Input Validation
Use Form Request classes for validation:
- `UpdateReviewDraftRequest` - Review draft updates
- `BulkDeleteRequest` - Bulk delete operations

## Code Style & Conventions

- Uses Laravel Pint for code formatting (run with `./vendor/bin/pint`)
- Follow Laravel conventions for naming (StudlyCase for classes, snake_case for database)
- Use type hints in all new code (PHP 8.2+ features)
- Service classes for business logic, Controllers stay thin

## Security Audit

A comprehensive security audit was performed on 2025-11-22. All critical and high-severity issues have been resolved. See `ISSUES.md` for the complete audit report and remediation details.

