# Auto-Ingestion Interval Settings - Feature Documentation

**Implementation Date:** 2025-10-22
**Feature Type:** User-Configurable System Settings
**Access Level:** Super Admin, Tenant Admin

---

## Table of Contents

1. [Feature Overview](#feature-overview)
2. [Access Control & Permissions](#access-control--permissions)
3. [Implementation Architecture](#implementation-architecture)
4. [How It Works](#how-it-works)
5. [Configuration Options](#configuration-options)
6. [Technical Implementation](#technical-implementation)
7. [File Structure](#file-structure)
8. [Database Schema](#database-schema)
9. [Testing Guide](#testing-guide)
10. [Troubleshooting](#troubleshooting)

---

## Feature Overview

### Purpose
Allow administrators to control how frequently the system automatically fetches new emails and reviews from external sources (Gmail API, IMAP servers, Google Business Profile).

### Key Benefits
- **Customizable intervals**: 5, 10, 15, 30, 45, 60 minutes, or custom (1-1440 minutes)
- **Per-tenant configuration**: Each tenant can set their own ingestion frequency
- **Resource optimization**: Reduces unnecessary API calls based on business needs
- **Cost management**: Lower API usage for tenants with less frequent needs

### Before This Feature
- ❌ Hardcoded 15-minute intervals for all tenants
- ❌ No way to change frequency without code modification
- ❌ All tenants had same ingestion schedule

### After This Feature
- ✅ User-friendly dropdown interface
- ✅ Custom interval input (1-1440 minutes)
- ✅ Per-tenant configuration
- ✅ Separate settings for emails and reviews

---

## Access Control & Permissions

### Role-Based Access (Enterprise Standard)

#### Super Admin
- **Access**: ✅ Full access to settings
- **Scope**: Can configure global default intervals
- **Route**: `/super-admin/settings`
- **Affects**: All tenants (by default, using first tenant's settings)

#### Tenant Admin
- **Access**: ✅ Full access to settings
- **Scope**: Can configure intervals for their own tenant only
- **Route**: `/tenant-admin/settings`
- **Affects**: Only their organization

#### Agent
- **Access**: ❌ No access (following principle of least privilege)
- **Reason**: Agents should not modify system configuration
- **Alternative**: Tenant Admin must configure for the organization

---

## Implementation Architecture

### High-Level Flow

```
┌─────────────────────────────────────────────────────────────┐
│ User Interface (Super Admin / Tenant Admin)                 │
│ - Dropdown: 5, 10, 15, 30, 45, 60 min, Custom              │
│ - Custom input: 1-1440 minutes                              │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ Settings Controller                                          │
│ - Validates input                                            │
│ - Saves to tenant.settings JSON column                       │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ Database: tenants.settings (JSON)                           │
│ {                                                            │
│   "email_ingestion_interval": 15,                           │
│   "review_ingestion_interval": 60                           │
│ }                                                            │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ Laravel Scheduler (routes/console.php)                      │
│ - Runs every minute                                          │
│ - Calls: emails:ingest-all                                   │
│ - Calls: reviews:ingest-all                                  │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ Ingestion Commands (Check Intervals)                        │
│ - Groups accounts/locations by tenant                        │
│ - Reads tenant.settings[interval]                            │
│ - Checks cache for last run time                             │
│ - If interval elapsed: Dispatch jobs                         │
│ - If not: Skip tenant                                        │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ Queue Jobs                                                   │
│ - IngestEmailsJob: Fetches emails for account               │
│ - IngestReviewsJob: Fetches reviews for location            │
└─────────────────────────────────────────────────────────────┘
```

---

## How It Works

### Step-by-Step Process

#### 1. Admin Configures Interval

**Super Admin:**
```
1. Navigates to /super-admin/settings
2. Selects "Email Auto-Ingestion Interval": 10 minutes
3. Selects "Review Auto-Ingestion Interval": 30 minutes
4. Clicks "Save Settings"
5. Settings stored in first tenant's settings JSON
```

**Tenant Admin:**
```
1. Navigates to /tenant-admin/settings
2. Selects "Email Auto-Ingestion Interval": 5 minutes
3. Selects "Review Auto-Ingestion Interval": 1 hour
4. Clicks "Save Settings"
5. Settings stored in their tenant's settings JSON
```

#### 2. Scheduler Runs Every Minute

```php
// routes/console.php (Line 17-22)
Schedule::command('emails:ingest-all')
    ->everyMinute()  // Changed from everyFifteenMinutes()
    ->name('email-auto-ingest')
    ->withoutOverlapping();
```

#### 3. Command Checks Tenant Intervals

```php
// app/Console/Commands/IngestAllEmailsCommand.php

// Get tenant's configured interval (default: 15 minutes)
$interval = $tenant->settings['email_ingestion_interval'] ?? 15;

// Check cache for last run time
$cacheKey = "email_ingestion_last_run_tenant_{$tenantId}";
$lastRun = Cache::get($cacheKey);

// Skip if interval not reached
if ($lastRun && now()->diffInMinutes($lastRun) < $interval) {
    // Skip - not enough time has passed
    continue;
}

// Interval reached - dispatch jobs
IngestEmailsJob::dispatch($mailAccount);

// Update cache with current timestamp
Cache::put($cacheKey, now(), now()->addDay());
```

#### 4. Jobs Execute

- Email ingestion runs every X minutes (per tenant configuration)
- Review ingestion runs every Y minutes (per tenant configuration)

---

## Configuration Options

### Available Intervals

| Dropdown Option | Value (minutes) | Use Case |
|----------------|-----------------|----------|
| 5 minutes      | 5               | High-volume businesses requiring real-time monitoring |
| 10 minutes     | 10              | Active businesses with frequent customer interactions |
| **15 minutes** | **15**          | **Default for emails** - Balanced approach |
| 30 minutes     | 30              | Moderate-volume businesses |
| 45 minutes     | 45              | Lower-volume businesses |
| **1 hour**     | **60**          | **Default for reviews** - Standard monitoring |
| Custom         | 1-1440          | Specific business requirements |

### Custom Interval Input

- **Minimum**: 1 minute (not recommended - high API usage)
- **Maximum**: 1440 minutes (24 hours)
- **Validation**: Integer between 1-1440
- **Example**: Set to 90 minutes (1.5 hours)

---

## Technical Implementation

### Controllers

#### Super Admin Settings Controller

**File**: `app/Http/Controllers/SuperAdmin/SettingsController.php`

```php
namespace App\Http\Controllers\SuperAdmin;

class SettingsController extends Controller
{
    public function index()
    {
        return view('super-admin.settings.index');
    }

    public function updateIngestionInterval(Request $request)
    {
        $request->validate([
            'email_interval' => 'required|string',
            'email_custom_interval' => 'nullable|integer|min:1|max:1440',
            'review_interval' => 'required|string',
            'review_custom_interval' => 'nullable|integer|min:1|max:1440',
        ]);

        // Determine final interval values
        $emailInterval = $request->email_interval === 'custom'
            ? $request->email_custom_interval
            : (int) $request->email_interval;

        $reviewInterval = $request->review_interval === 'custom'
            ? $request->review_custom_interval
            : (int) $request->review_interval;

        // Save to tenant settings
        $defaultTenant = Tenant::first();
        if ($defaultTenant) {
            $settings = $defaultTenant->settings ?? [];
            $settings['email_ingestion_interval'] = $emailInterval;
            $settings['review_ingestion_interval'] = $reviewInterval;
            $defaultTenant->update(['settings' => $settings]);
        }

        return redirect()
            ->route('super-admin.settings.index')
            ->with('success', 'Settings updated successfully!');
    }
}
```

#### Tenant Admin Settings Controller

**File**: `app/Http/Controllers/TenantAdmin/SettingsController.php`

- Same implementation as Super Admin
- Uses `auth()->user()->tenant` instead of `Tenant::first()`
- Only affects the logged-in user's tenant

### Routes

**File**: `routes/web.php`

```php
// Super Admin routes (Line 90-92)
Route::get('/settings', [App\Http\Controllers\SuperAdmin\SettingsController::class, 'index'])
    ->name('settings.index');
Route::post('/settings/ingestion-interval', [App\Http\Controllers\SuperAdmin\SettingsController::class, 'updateIngestionInterval'])
    ->name('settings.update-ingestion-interval');

// Tenant Admin routes (Line 119-121)
Route::get('/settings', [App\Http\Controllers\TenantAdmin\SettingsController::class, 'index'])
    ->name('settings.index');
Route::post('/settings/ingestion-interval', [App\Http\Controllers\TenantAdmin\SettingsController::class, 'updateIngestionInterval'])
    ->name('settings.update-ingestion-interval');
```

### Views

#### Super Admin Settings View

**File**: `resources/views/super-admin/settings/index.blade.php`

**Key Features**:
- Dropdown for email interval
- Dropdown for review interval
- Custom input field (hidden by default)
- JavaScript to toggle custom input
- Form submission to save settings
- Success/error flash messages

**JavaScript Toggle Function**:
```javascript
function toggleCustomInput(type) {
    const select = document.getElementById(`${type}_interval`);
    const customInput = document.getElementById(`${type}_custom_input`);

    if (select.value === 'custom') {
        customInput.classList.remove('hidden');
    } else {
        customInput.classList.add('hidden');
    }
}
```

#### Tenant Admin Settings View

**File**: `resources/views/tenant-admin/settings/index.blade.php`

- Identical to Super Admin view
- Navigation includes tenant-admin specific links
- Description mentions "for your organization"

### Scheduler Configuration

**File**: `routes/console.php`

**Before:**
```php
Schedule::command('emails:ingest-all')
    ->everyFifteenMinutes()  // Hardcoded
```

**After:**
```php
Schedule::command('emails:ingest-all')
    ->everyMinute()  // Checks tenant settings internally
```

### Ingestion Commands

#### Email Ingestion Command

**File**: `app/Console/Commands/IngestAllEmailsCommand.php`

**Changes Made** (Lines 45-95):
1. Load mail accounts with `user.tenant` relationship
2. Group accounts by `tenant_id`
3. For each tenant:
   - Read `email_ingestion_interval` from settings (default: 15)
   - Check cache key: `email_ingestion_last_run_tenant_{$tenantId}`
   - If interval elapsed: dispatch jobs, update cache
   - If not: skip tenant

**Cache Strategy**:
- **Key**: `email_ingestion_last_run_tenant_{tenantId}`
- **Value**: Current timestamp
- **TTL**: 1 day (cache expires after 24 hours)

#### Review Ingestion Command

**File**: `app/Console/Commands/IngestAllReviewsCommand.php`

**Changes Made** (Lines 45-95):
- Same logic as email ingestion
- Reads `review_ingestion_interval` instead
- Default: 60 minutes
- Cache key: `review_ingestion_last_run_tenant_{tenantId}`

---

## File Structure

```
replypilot/
├── app/
│   ├── Console/
│   │   └── Commands/
│   │       ├── IngestAllEmailsCommand.php       ✅ MODIFIED
│   │       └── IngestAllReviewsCommand.php      ✅ MODIFIED
│   └── Http/
│       └── Controllers/
│           ├── SuperAdmin/
│           │   └── SettingsController.php       ✅ NEW
│           └── TenantAdmin/
│               └── SettingsController.php       ✅ NEW
├── resources/
│   └── views/
│       ├── super-admin/
│       │   ├── email/
│       │   │   └── index.blade.php              ✅ MODIFIED (navigation)
│       │   └── settings/
│       │       └── index.blade.php              ✅ NEW
│       └── tenant-admin/
│           ├── email/
│           │   └── index.blade.php              ✅ MODIFIED (navigation)
│           └── settings/
│               └── index.blade.php              ✅ NEW
├── routes/
│   ├── console.php                              ✅ MODIFIED
│   └── web.php                                  ✅ MODIFIED
└── AUTO_INGESTION_INTERVAL_SETTINGS.md          ✅ NEW (this file)
```

---

## Database Schema

### Storage Location

**Table**: `tenants`
**Column**: `settings` (JSON type)

### Schema Structure

```json
{
  "email_ingestion_interval": 15,
  "review_ingestion_interval": 60
}
```

### Default Values

| Setting | Default | Notes |
|---------|---------|-------|
| `email_ingestion_interval` | 15 | Used if not set in settings |
| `review_ingestion_interval` | 60 | Used if not set in settings |

### Example Queries

**Get tenant's email interval:**
```php
$interval = $tenant->settings['email_ingestion_interval'] ?? 15;
```

**Update tenant's settings:**
```php
$settings = $tenant->settings ?? [];
$settings['email_ingestion_interval'] = 30;
$tenant->update(['settings' => $settings]);
```

---

## Testing Guide

### Manual Testing Steps

#### Test as Super Admin

1. **Login as Super Admin**
   ```
   Email: admin@example.com (or your super admin)
   ```

2. **Navigate to Settings**
   ```
   URL: http://127.0.0.1:8000/super-admin/settings
   ```

3. **Test Preset Intervals**
   - Select "Email Auto-Ingestion Interval": 5 minutes
   - Select "Review Auto-Ingestion Interval": 30 minutes
   - Click "Save Settings"
   - Verify success message appears

4. **Test Custom Interval**
   - Select "Email Auto-Ingestion Interval": Custom
   - Enter: 90 (minutes)
   - Click "Save Settings"
   - Verify custom value is saved

5. **Verify Database**
   ```sql
   SELECT id, name, settings FROM tenants;
   ```
   Expected:
   ```json
   {
     "email_ingestion_interval": 90,
     "review_ingestion_interval": 30
   }
   ```

#### Test as Tenant Admin

1. **Login as Tenant Admin**

2. **Navigate to Settings**
   ```
   URL: http://127.0.0.1:8000/tenant-admin/settings
   ```

3. **Configure Intervals**
   - Set different values than Super Admin
   - Verify only your tenant's settings are updated

4. **Verify Isolation**
   - Super Admin's tenant should have different settings
   - Tenant Admin's tenant should have their own settings

#### Test Scheduler Behavior

1. **Start Scheduler**
   ```bash
   php artisan schedule:work
   ```

2. **Set Short Interval** (e.g., 2 minutes)

3. **Watch Logs**
   ```bash
   tail -f storage/logs/email-ingestion.log
   ```

4. **Verify Behavior**
   - Command runs every minute
   - Jobs dispatched when interval reached
   - Tenants skipped when interval not reached

### Expected Log Output

```
[2025-10-22 10:00:00] Starting email ingestion for all accounts...
[2025-10-22 10:00:00] Found 5 mail account(s). Checking tenant intervals...
[2025-10-22 10:00:00] ⏭ Skipping tenant Default Business: interval not reached (15 min)
[2025-10-22 10:00:00] Dispatched 0 job(s), skipped 5 (interval not reached).

... (14 minutes later)

[2025-10-22 10:15:00] Starting email ingestion for all accounts...
[2025-10-22 10:15:00] Found 5 mail account(s). Checking tenant intervals...
[2025-10-22 10:15:00] ✓ Dispatched job for: test@example.com
[2025-10-22 10:15:00] Dispatched 5 job(s), skipped 0 (interval not reached).
```

### Automated Testing

```bash
# Test scheduler status
php artisan schedule:list

# Test ingestion commands manually
php artisan emails:ingest-all
php artisan reviews:ingest-all

# Check cache values
php artisan tinker
>>> Cache::get('email_ingestion_last_run_tenant_1');
```

---

## Troubleshooting

### Issue: Interval Not Being Respected

**Symptoms:**
- Jobs still running every 15 minutes despite changing settings
- Logs show all jobs being dispatched

**Possible Causes & Solutions:**

1. **Cache Not Cleared**
   ```bash
   php artisan cache:clear
   ```

2. **Old Scheduler Still Running**
   - Stop `php artisan schedule:work`
   - Restart: `php artisan schedule:work`

3. **Settings Not Saved**
   ```sql
   SELECT settings FROM tenants WHERE id = 1;
   ```
   - Verify JSON contains correct values

### Issue: Custom Interval Not Showing

**Symptoms:**
- Custom input field doesn't appear
- JavaScript error in console

**Solution:**
- Check browser console for JavaScript errors
- Verify `toggleCustomInput()` function is defined
- Hard refresh: Ctrl+F5 (Windows) or Cmd+Shift+R (Mac)

### Issue: Settings Page 404

**Symptoms:**
- `/super-admin/settings` returns 404
- Route not found error

**Solution:**
1. **Clear route cache**
   ```bash
   php artisan route:clear
   php artisan route:cache
   ```

2. **Verify routes registered**
   ```bash
   php artisan route:list | grep settings
   ```

### Issue: Unauthorized Access

**Symptoms:**
- Tenant Admin can't access settings
- Middleware blocking access

**Solution:**
- Verify middleware in `routes/web.php`
- Check user role: `auth()->user()->role`
- Ensure user is `tenant-admin` role

---

## Best Practices

### Recommended Intervals

| Business Type | Email Interval | Review Interval | Reason |
|--------------|----------------|-----------------|---------|
| High-volume e-commerce | 5-10 min | 30-60 min | Fast customer response needed |
| Service business | 15-30 min | 60 min | Balanced monitoring |
| Low-volume retail | 30-60 min | 2-4 hours | Cost optimization |
| Enterprise | Custom | Custom | Based on SLA requirements |

### Performance Considerations

**Minimum Interval Guidelines:**
- **< 5 minutes**: High API usage, only for critical needs
- **5-15 minutes**: Standard for active businesses
- **15-60 minutes**: Balanced approach (recommended)
- **> 60 minutes**: For low-priority monitoring

**API Rate Limits:**
- Gmail API: 250 quota units/second per user
- Google Business Profile: Varies by API endpoint
- Consider API costs when setting very short intervals

---

## Future Enhancements

### Potential Improvements

1. **Per-Account Intervals**
   - Allow different intervals for different email accounts
   - High-priority accounts: 5 min, Others: 30 min

2. **Time-Based Schedules**
   - Business hours: Every 10 minutes
   - After hours: Every 60 minutes

3. **API Usage Dashboard**
   - Show API calls per day
   - Cost estimation based on intervals

4. **Smart Intervals**
   - Auto-adjust based on email volume
   - Increase frequency when high volume detected

5. **Notification Thresholds**
   - Alert when X new emails detected
   - Trigger immediate ingestion on threshold

---

## Related Documentation

- **AUTO_INGESTION.md**: Overview of auto-ingestion system
- **SESSION_HISTORY.md**: Development history and changes
- **CLAUDE.md**: Project structure and commands

---

**Document Version:** 1.0
**Last Updated:** 2025-10-22
**Maintained By:** Development Team
**Implementation Status:** ✅ Complete and Production-Ready
