# AI Provider Setup Guide

ReplyPilot now supports multiple AI providers for generating email and review replies. You can configure which providers to use and set a default provider.

## Supported AI Providers

### 1. Google Gemini (Default)
- **Provider**: Google Gemini
- **Model**: gemini-1.5-flash
- **Status**: Default provider
- **API**: Google AI Studio API

### 2. OpenAI ChatGPT
- **Provider**: OpenAI
- **Model**: gpt-4o-mini
- **Status**: Optional
- **API**: OpenAI API

### 3. Anthropic Claude
- **Provider**: Anthropic
- **Model**: claude-3-haiku-20240307
- **Status**: Optional
- **API**: Anthropic API

## Environment Configuration

Add the following to your `.env` file:

```env
# AI Provider Configuration
AI_DEFAULT_PROVIDER=gemini
AI_GEMINI_ENABLED=true
AI_OPENAI_ENABLED=false
AI_ANTHROPIC_ENABLED=false

# API Keys
GEMINI_API_KEY=your_gemini_api_key_here
OPENAI_API_KEY=your_openai_api_key_here
ANTHROPIC_API_KEY=your_anthropic_api_key_here
```

## Getting API Keys

### Google Gemini API Key
1. Go to [Google AI Studio](https://makersuite.google.com/app/apikey)
2. Sign in with your Google account
3. Click "Create API Key"
4. Copy the generated API key
5. Add it to your `.env` file as `GEMINI_API_KEY`

### OpenAI API Key
1. Go to [OpenAI Platform](https://platform.openai.com/api-keys)
2. Sign in to your OpenAI account
3. Click "Create new secret key"
4. Copy the generated API key
5. Add it to your `.env` file as `OPENAI_API_KEY`

### Anthropic API Key
1. Go to [Anthropic Console](https://console.anthropic.com/)
2. Sign in to your Anthropic account
3. Navigate to API Keys section
4. Click "Create Key"
5. Copy the generated API key
6. Add it to your `.env` file as `ANTHROPIC_API_KEY`

## Configuration Options

### Provider Settings
- `AI_DEFAULT_PROVIDER`: Set the default AI provider (gemini, openai, anthropic)
- `AI_GEMINI_ENABLED`: Enable/disable Gemini provider (true/false)
- `AI_OPENAI_ENABLED`: Enable/disable OpenAI provider (true/false)
- `AI_ANTHROPIC_ENABLED`: Enable/disable Anthropic provider (true/false)

### Model Settings
- `GEMINI_MODEL`: Gemini model to use (default: gemini-1.5-flash)
- `OPENAI_MODEL`: OpenAI model to use (default: gpt-4o-mini)
- `ANTHROPIC_MODEL`: Anthropic model to use (default: claude-3-haiku-20240307)

## Usage

### Web Interface
1. Navigate to "AI Settings" in the main navigation
2. View provider status and configuration
3. Test individual provider connections
4. Test all providers at once
5. Enable/disable providers as needed

### Programmatic Usage
```php
use App\Services\AIServiceManager;
use App\Services\EnhancedReplyDraftService;

// Get AI manager
$aiManager = app(AIServiceManager::class);

// Get default provider
$defaultProvider = $aiManager->getDefaultProvider();

// Get specific provider
$geminiProvider = $aiManager->getProvider('gemini');

// Use enhanced reply service
$replyService = new EnhancedReplyDraftService($aiManager, 'gemini');
$result = $replyService->makeForReview($review);
```

## Testing Connections

### Via Web Interface
1. Go to AI Settings page
2. Click "Test Connection" for individual providers
3. Click "Test All Providers" to test all enabled providers

### Via Command Line
```bash
# Test all providers
php artisan tinker
>>> app(App\Services\AIServiceManager::class)->testAllProviders()
```

## Troubleshooting

### Common Issues

1. **"Provider not available" error**
   - Check if the provider is enabled in `.env`
   - Verify the API key is correctly set
   - Ensure the API key has proper permissions

2. **"Connection failed" error**
   - Verify API key is valid and active
   - Check internet connectivity
   - Ensure API quotas haven't been exceeded

3. **"No AI providers are available" error**
   - At least one provider must be enabled
   - Check `.env` configuration
   - Verify API keys are set

### Debug Mode
Enable debug logging by setting `LOG_LEVEL=debug` in your `.env` file to see detailed error messages.

## Migration from Single Provider

If you're upgrading from the original single-provider setup:

1. The original `ReplyDraftService` remains unchanged and functional
2. The new `EnhancedReplyDraftService` provides multi-provider support
3. Existing functionality continues to work without changes
4. Gradually migrate to the enhanced service as needed

## Performance Considerations

- **Gemini**: Fastest response times, good for high-volume usage
- **OpenAI**: Reliable performance, widely supported
- **Anthropic**: High-quality responses, good for complex tasks

Choose providers based on your specific needs for speed, quality, and cost.
